# -*- coding: utf-8 -*-
import datetime
import json
import os
import shutil
import uuid
import logging

from django.conf import settings
from django.contrib.auth.models import (
    User,
    Group,
    Permission,
    ContentType,
)
from django_docopt_command import DocOptCommand

import yenv

from passport_grants_configurator.apps.core.exceptions import ExportError, ProcessError
from passport_grants_configurator.apps.core.utils import (
    deb_changelog,
    git_diff,
    git_pull,
    git_push,
    run_exclusively,
)
from passport_grants_configurator.apps.passport_admin_roles_adapter.models import Timestamp

logger = logging.getLogger(__name__)

GIT_API = settings.PASSPORT_ADMIN_ROLES_GIT_API


class Command(DocOptCommand):
    docs = '''Usage: export_roles --settings=<settings>'''

    def serialize(self):
        """
        Подготавливаем к сериализации Пользователей, Группы пользователей(роли) и Разрешения
        :return: Словарь со списками объектов
        """
        users = User.objects.prefetch_related('groups').order_by('username')
        meta_users = [
            {
                'username': user.username,
                'roles': list(user.groups.values_list('id', flat=True).order_by('id')),
            }
            for user in users
        ]

        groups = Group.objects.prefetch_related('permissions').order_by('id')
        meta_groups = [
            {
                'id': group.id,
                'name': group.name,
                'grants': list(group.permissions.values_list('id', flat=True).order_by('id')),
            }
            for group in groups
        ]

        # использую специальное имя pa_permission,
        # которым помечены все гранты для Паспортной админки
        pa_permission = ContentType.objects.get(name='pa_permission')
        grants = Permission.objects.filter(content_type_id=pa_permission.id).order_by('id')

        meta_grants = [
            {
                'id': grant.id,
                'name': grant.codename,
                'description': grant.name,
            }
            for grant in grants
        ]

        return {
            'users': meta_users,
            'roles': meta_groups,
            'grants': meta_grants,
        }

    def dump_json(self, document, filename):
        """
        Сохраним в файл читабельную версию информации о всех пользователях, ролях, правах
        :param document: объект со списками объектов
        :param filename: имя файла для сохранения
        """
        with open(filename, 'w') as fout:
            json.dump(document, fout, sort_keys=True, indent=4)

    @run_exclusively('/passport_grants_configurator/management/export_roles', logger)
    def handle_docopt(self, arguments):
        last_export = Timestamp.last_export()
        time_snap = last_export.time

        if not Timestamp.there_are_unexported_changes():
            logger.debug(
                'Need no action right now. Last export was at %s, last change at %s',
                time_snap,
                Timestamp.last_change().time,
            )
            return

        logger.info('Start export passport-admin-roles')
        repository_dir = os.path.join(GIT_API['working_dir'], str(uuid.uuid4()))
        os.makedirs(repository_dir)

        try:
            git_pull(
                repository=GIT_API['repo'],
                repository_dir=repository_dir,
            )

            doc = self.serialize()

            filename = os.path.join(repository_dir, 'roles/roles.%s.json' % yenv.type)
            self.dump_json(doc, filename)

            # девелоперские роли используем и в тестинге и наоборот
            if yenv.type == 'testing':
                filename = os.path.join(repository_dir, 'roles/roles.development.json')
                self.dump_json(doc, filename)

            # Выходим, если ничего не изменилось
            if not git_diff(repository_dir):
                logger.debug('No changes detected')
                return

            last_export.time = datetime.datetime.now()
            last_export.save()

            deb_changelog(
                repository_dir=repository_dir,
                committer=GIT_API['committer'],
                message='roles changed',
            )

            git_push(
                repository=GIT_API['repo'],
                repository_dir=repository_dir,
                committer=GIT_API['committer'],
                message='roles changed',
            )

        except (ProcessError, ExportError):
            logger.exception('Export roles failed')
            last_export.time = time_snap
            last_export.save()
            return

        finally:
            shutil.rmtree(repository_dir)
