import argparse

import argcomplete
from passport.backend.utils.text import kebab_to_snake


class ArgumentGroup(object):
    def __init__(self, *args, **kwargs):
        self.chain = []
        if args or kwargs:
            self.add_argument(*args, **kwargs)

    def add_argument(self, *args, **kwargs):
        self.chain.append({'args': args, 'kwargs': kwargs})
        return self


class Commander(object):
    def __init__(
        self,
        parser=None,
        processor=None,
        name='root',
        expand_kwargs=True,
        args_set=None,
        description=None,
        disable_argcomplete=False,
        *args,
        **kwargs
    ):
        if not parser:
            parser = argparse.ArgumentParser(description=description)
        self.parser = parser
        self.subparsers = None
        self.subcommands = {}
        self.processor = processor
        self.name = name
        self.expand_kwargs = expand_kwargs
        self.args = args
        self.kwargs = kwargs
        self.args_set = args_set
        self.disable_argcomplete = disable_argcomplete
        if self.args_set is None:
            self.args_set = set()

    def add_command(self, name, processor=None, expand_kwargs=True, args_set=None, required=True, *args, **kwargs):
        if not self.subparsers:
            self.subparsers = self.parser.add_subparsers(dest=self.name)
            self.subparsers.required = required  # не передаём параметром в add_subparsers, так как py2 этого не умеет

        # объединяем возможные параметры родительской команды с параметрами текущей
        args_set = self.args_set | (args_set or set())

        subparser = self.subparsers.add_parser(name)
        subcommand = Commander(
            parser=subparser,
            processor=processor,
            name=name,
            expand_kwargs=expand_kwargs,
            args_set=args_set,
            *args,
            **kwargs
        )
        self.subcommands[name] = subcommand
        return subcommand

    def add_argument(self, *args, **kwargs):
        if args and isinstance(args[0], ArgumentGroup):
            self.add_group(args[0])
            return self
        if 'dest' in kwargs:
            if kwargs['dest'] in self.args_set:
                return self
            self.args_set.add(kwargs['dest'])
        elif len(args) > 0:
            dest = args[0].lstrip('-').replace('-', '_')  # https://docs.python.org/3/library/argparse.html#dest
            if dest in self.args_set:
                return self
            self.args_set.add(dest)
        self.parser.add_argument(*args, **kwargs)
        return self

    def add_group(self, group):
        for record in group.chain:
            self.add_argument(*record['args'], **record['kwargs'])
        return self

    def add_mutually_exclusive_group(self):
        return Commander(parser=self.parser.add_mutually_exclusive_group(), args_set=self.args_set)

    def invoke(self, parsed_args=None):
        if not self.disable_argcomplete:
            argcomplete.autocomplete(self.parser)

        if parsed_args is None:
            parsed_args = self.parser.parse_args()

        if getattr(parsed_args, self.name, None) in self.subcommands:
            return self.subcommands[getattr(parsed_args, self.name)].invoke(
                parsed_args=parsed_args,
            )
        if self.processor:
            if self.expand_kwargs:
                kwargs = {kebab_to_snake(k): v for k, v in vars(parsed_args).items() if k in self.args_set}
                kwargs.update(self.kwargs)
                return self.processor(*self.args, **kwargs)
            else:
                return self.processor(parsed_args=parsed_args, *self.args, **self.kwargs)
        else:
            if self.expand_kwargs:
                return {k: v for k, v in vars(parsed_args).items() if k in self.args_set}
            else:
                return parsed_args
