from collections import namedtuple
import json
import logging.config
import os
import os.path
from types import ModuleType

from jinja2 import Template
from passport.backend.utils.common import (
    smart_merge_dicts,
    unflatten,
)
from passport.backend.utils.file import (
    mkdir_p,
    read_file,
)
import six
from six import string_types
import yaml


LOGPATH_OPTION_NAME = 'logpath'


class Configurator(dict):
    def __init__(self, name, configs, encoding=None, sources_list=None):
        """
        Create Configurator object

        :param name: name used for looking in /etc/yandex folder
        :param configs: list of paths to config files, dicts or modules.
        End filepath with `?` to mark it as optional config.
        """
        super(Configurator, self).__init__()

        self.name = name
        self.encoding = encoding
        self.sources_list = sources_list
        self._by_basenames = {}
        self._omitted_files = []

        for config in configs:
            file_found = self.update(config)
            if not file_found:
                self._omitted_files.append(config)

        overriden_options = os.environ.get('CONFIGURATOR')
        if overriden_options:
            overriden_options = dict(map(lambda x: x.split('='), overriden_options.split(';')))
            self.update(unflatten(overriden_options))

    def _config_file_name(self, filename):
        filename = os.path.expanduser(filename)
        if os.path.isabs(filename) or filename.startswith('./'):
            return filename
        else:
            return os.path.join('/etc/yandex/', self.name, filename)

    def update(self, new_config, basename=None, **kwargs):
        if isinstance(new_config, string_types):
            optional = new_config.endswith('?')
            filename = new_config.rstrip('?')
            basename = basename or os.path.basename(filename)

            try:
                data = read_file(
                    self._config_file_name(filename),
                    sources_list=self.sources_list,
                    encoding=self.encoding,
                )
            except IOError as e:
                if optional:
                    return False
                else:
                    raise e

            if six.PY3 and isinstance(data, bytes):
                data = data.decode('utf-8')
            data = Template(data).render(**self)

            if filename.endswith('.json'):
                new_config = json.loads(data)
            elif filename.endswith('.yaml'):
                new_config = yaml.safe_load(data)

        elif isinstance(new_config, ModuleType):
            new_config = new_config.__dict__

        elif callable(new_config):
            new_config = new_config(self)

        if not new_config:
            new_config = {}

        for k in new_config:
            if callable(new_config[k]):
                new_config[k] = new_config[k](context=self)

        if LOGPATH_OPTION_NAME in new_config:
            new_config[LOGPATH_OPTION_NAME] = os.path.expanduser(new_config[LOGPATH_OPTION_NAME]).rstrip('/')

        smart_merge_dicts(self, new_config, list_policy='override', copy=False)
        if basename:
            self._by_basenames[basename] = new_config

        return True

    def set_as_passport_settings(self, basename_to_export='export.yaml'):
        from passport.backend.core.conf import settings as passport_settings
        if not passport_settings.configured and basename_to_export in self._by_basenames:
            if 'HOSTS' in self._by_basenames[basename_to_export]:
                Host = namedtuple('Host', ['name', 'id', 'dc'])
                self._by_basenames[basename_to_export]['HOSTS'] = [
                    (Host(**host) if isinstance(host, dict) else host)
                    for host in self._by_basenames[basename_to_export]['HOSTS']
                ]
            passport_settings.configure(**self.get_config_by_basename(basename_to_export))

    def get_config_by_basename(self, basename):
        return self._by_basenames[basename]

    def set_logging(self, make_path=False):
        if make_path:
            mkdir_p(self[LOGPATH_OPTION_NAME])
        logging.config.dictConfig(self['logging'])

    def has_missed_configs(self):
        return bool(self._omitted_files)

    def has(self, *args):
        current = self
        for c in args:
            if c not in current:
                return False
            current = current[c]
        return True
