import heapq
import random
import time

from ..threading import PortableRWLock as RWLock


class AlreadyExistsException(Exception):
    pass


class NoAvailableHostsException(Exception):
    pass


class HostManager(object):
    def __init__(self, threaded=False):
        self._hosts_heap = []
        self._hosts_set = set()
        self._hosts_priorities = {}
        self._cooldown_list = []
        self._cooldown_set = set()
        self._hosts_lock = RWLock()
        self._threaded = threaded

    def add(self, priority, host):
        """
        :param priority: priorities are inverted, i.e a host with priority 1
        is higher than another host with priority 2.
        :param host:
        :return: None
        """
        if self._threaded:
            self._hosts_lock.acquire_write()
        try:
            if host in self._hosts_set or host in self._cooldown_set:
                raise AlreadyExistsException()
            self._hosts_set.add(host)
            self._hosts_priorities[host] = priority

            for stored_priority, stored_hosts in self._hosts_heap:
                if stored_priority == priority:
                    stored_hosts.append(host)
                    return

            heapq.heappush(self._hosts_heap, (priority, [host]))
        finally:
            if self._threaded:
                self._hosts_lock.release_write()

    def cooldown(self, host, cooldown_time):
        """
        Remove host from host manager for cooldown_time
        :param host:
        :param cooldown_time:
        :return:
        """
        if self._threaded:
            self._hosts_lock.acquire_write()
        try:
            if host not in self._hosts_set or host in self._cooldown_set:
                return
            priority = self._hosts_priorities[host]
            for stored_priority, stored_hosts in self._hosts_heap:
                if stored_priority == priority:
                    self._cooldown_set.add(host)

                    stored_hosts.remove(host)
                    if len(stored_hosts) == 0:
                        self._hosts_heap.remove((stored_priority, stored_hosts))
                        heapq.heapify(self._hosts_heap)

                    heapq.heappush(
                        self._cooldown_list,
                        (-(time.time() + cooldown_time), (priority, host))
                    )
                    return
        finally:
            if self._threaded:
                self._hosts_lock.release_write()

    def reborn(self):
        """
        Returns host from cooldown realm to the reality.
        Normally called from `get` function, there is no need
        to call explicitly.
        :return:
        """
        current_time = time.time()
        while (
            len(self._cooldown_list) > 0 and
            self._cooldown_list[0][0] > -current_time
        ):
            if self._threaded:
                self._hosts_lock.acquire_write()
            try:
                while (
                    len(self._cooldown_list) > 0 and
                    self._cooldown_list[0][0] > -current_time
                ):
                    priority, host = self._cooldown_list[0][1]

                    heapq.heappop(self._cooldown_list)

                    for stored_priority, stored_hosts in self._hosts_heap:
                        if stored_priority == priority:
                            stored_hosts.append(host)
                            self._cooldown_set.remove(host)
                            break

                    if host in self._cooldown_set:
                        heapq.heappush(self._hosts_heap, (priority, [host]))
                        self._cooldown_set.remove(host)
            finally:
                if self._threaded:
                    self._hosts_lock.release_write()

    def get(self):
        self.reborn()
        if self._threaded:
            self._hosts_lock.acquire_read()
        try:
            if len(self._hosts_heap) > 0:
                return random.choice(self._hosts_heap[0][1])
            else:
                raise NoAvailableHostsException(
                    'No available hosts among: %s' % str(sorted(self.hosts_set()))
                )
        finally:
            if self._threaded:
                self._hosts_lock.release_read()

    def hosts_set(self):
        return set(self._hosts_set)

    def __len__(self):
        return len(self._hosts_set)

    def alive(self):
        with self._hosts_lock:
            return len(self) - len(self._cooldown_set)
