from abc import (
    ABCMeta,
    abstractmethod,
)
import inspect


class Aggregator(object):
    __metaclass__ = ABCMeta

    def __init__(self, event, *fields):
        self.event = event
        self.fields = fields

    @abstractmethod
    def proceed(self, **fields):
        pass

    @abstractmethod
    def get_value(self):
        pass


class SumAggregator(Aggregator):
    def __init__(self, event, field):
        self.agg = 0
        self.event = event
        self.fields = [field]

    def proceed(self, **fields):
        val = fields.get(self.fields[0], 0)
        self.agg += val

    def get_value(self):
        return self.agg


class LastAggregator(Aggregator):
    def __init__(self, event, field):
        self.agg = None
        self.event = event
        self.fields = [field]

    def proceed(self, **fields):
        val = fields.get(self.fields[0], self.agg)
        self.agg = val

    def get_value(self):
        return self.agg


class CustomAggregator(Aggregator):
    def __init__(self, event, func, start_value=0, *fields):
        self.event = event
        self.func = func
        self.fields = fields
        self.agg = start_value

    def proceed(self, **fields):
        self.agg = self.func(self.agg, *[fields[x] for x in self.fields])

    def get_value(self):
        return self.agg


def last(event, field):
    return LastAggregator(event, field)


def sum(event, field):
    return SumAggregator(event, field)


def custom(event, func, start_value=0):
    arg_names = inspect.getargspec(func)[0][1:]
    return CustomAggregator(event, func, start_value, *arg_names)
