import numpy as np
from passport.backend.library.yalearn.structures import RunningStatistics

from .aggregators import AggregatorsHolder


class EventStream(object):
    """
    Abstraction for processing sequential pairs of events.
    This class deals is an informal tuple of - event, timestamp and extra data
    """
    def __init__(self, events=None, aggregators=None):
        """
        Create EventStream
        Parameters:
          events - list of events to process, left ones will be missed
          aggregators - functions for calculating extra statistics on EventStream
        """
        self.last_event = None
        self.last_timestamp = None
        self.transition = {}
        self.counters = {}
        self.events = events
        self.agg = AggregatorsHolder(aggregators)

    def add(self, event, timestamp, **fields):
        """
        Add event to EventStream
        Parameters:
          event - name of event
          timestamp
        """
        if self.events is not None and event not in self.events:
            return

        self.counters.setdefault(event, 0)
        self.counters[event] += 1

        self.agg.proceed_by_event(event, **fields)

        if self.last_event is not None and self.last_timestamp is not None:
            key = '%s@%s' % (self.last_event, event)
            if key not in self.transition:
                self.transition[key] = RunningStatistics()
            self.transition[key].push(timestamp - self.last_timestamp)
        self.last_event = event
        self.last_timestamp = timestamp

    def mean(self, first, second=None):
        return np.mean(self.diff(first, second))

    def median(self, first, second=None):
        return np.median(self.diff(first, second))

    def std(self, first, second=None):
        return np.std(self.diff(first, second))

    def stats(self, first, second=None, bins=None, prefix=None):
        """
        Return statistics for series of events
        Parameters:
          first - name of the first event
          second - name of the second event
          bins - split diff series of two events to bins
          prefix - prefix for keys in result dictionary
        Return:
          dict
        """
        if second is None:
            second = first
        key = '%s@%s' % (first, second)
        if prefix is None:
            prefix = key + '_'
        len_name = prefix + 'len'
        if key not in self.transition:
            return {
                len_name: 0
            }
        running_stats = self.transition[key]
        result = {
            len_name: running_stats.size()
        }
        if result[len_name] == 0:
            return result

        sum_name = prefix + 'sum'
        max_name = prefix + 'max'
        min_name = prefix + 'min'
        mean_name = prefix + 'mean'
        std_name = prefix + 'std'
        skewness_name = prefix + 'skewness'
        kurtosis_name = prefix + 'kurtosis'

        result[sum_name] = running_stats.sum()
        result[max_name] = running_stats.max()
        result[min_name] = running_stats.min()
        result[mean_name] = running_stats.mean()
        result[std_name] = running_stats.std()
        result[skewness_name] = running_stats.skewness()
        result[kurtosis_name] = running_stats.kurtosis()
        return result

    def count(self, name):
        return self.counters.get(name, 0)
