# -*- coding: utf-8 -*-
import calendar
from datetime import (
    datetime,
    timedelta,
)
import json
import re

from passport.backend.core.crypto.exceptions import DecryptionError
from passport.backend.core.historydb.crypto import decrypt
from passport.backend.utils.string import smart_bytes


RFCTIME_RE = re.compile('^(\d{4})-(\d{2})-(\d{2}).(\d{2}):(\d{2}):(\d{2})\.(\d{6})([+-])(\d{2})$')


class JoinProcessors(object):
    def __init__(self, processors, stop_on=None):
        self.processors = processors
        self.stop_on = stop_on

    def __call__(self, fn):
        def wrapped(value):
            for processor in self.processors:
                value = processor(value)
                if value == self.stop_on:
                    return value
            return fn(value)
        return wrapped


def default_processor(value):
    if value in ['-']:
        return None
    return value


@JoinProcessors([default_processor])
def int_processor(value):
    try:
        return int(value)
    except (TypeError, ValueError):
        return None


@JoinProcessors([default_processor])
def float_processor(value):
    try:
        return float(value)
    except (TypeError, ValueError):
        return None


@JoinProcessors([default_processor])
def unhexify_processor(value):
    try:
        return int(value, 16)
    except (TypeError, ValueError):
        return None


@JoinProcessors([default_processor])
def ip_processor(value):
    if value.startswith('::ffff:'):
        return value.replace('::ffff:', '')
    return value


@JoinProcessors([default_processor])
def rfctime_processor(datetime_string):
    """Формат: 2012-06-05T12:33:58.407453+04"""
    match = RFCTIME_RE.match(datetime_string)
    if not match:
        return None

    offset_hour = int(match.group(9))
    if match.group(8) == '+':
        offset_hour = -offset_hour

    dt = datetime(*map(int, match.groups()[:-2])) + timedelta(hours=offset_hour)
    return calendar.timegm(dt.timetuple()) + dt.microsecond / 1e6


@JoinProcessors([default_processor])
def json_processor(string):
    try:
        return json.loads(string)
    except ValueError:
        return None


def decrypt_processor(encrypted_string):
    encrypted_string = smart_bytes(encrypted_string)
    try:
        return decrypt(encrypted_string)
    except DecryptionError:
        return
