# -*- coding: utf-8 -*-
import struct
import sys

from netaddr import (
    AddrFormatError,
    IPAddress,
)
from netaddr.strategy.ipv6 import packed_to_int as netaddr_packed_to_int
from passport.backend.core.geobase import (
    get_geobase,
    Region,
)
from passport.backend.core.portallib import (
    get_net,
    get_uatraits,
)


def pack_ip(ip):
    try:
        return IPAddress(ip).ipv6().packed
    except AddrFormatError:
        return


def pack_ip_shortest(ip):
    try:
        parsed_ip = IPAddress(ip)
        if parsed_ip.version == 4:
            return parsed_ip.ipv4().packed
        else:
            return parsed_ip.ipv6().packed
    except AddrFormatError:
        return


def unpack_ip(packed_ip):
    try:
        return IPAddress(netaddr_packed_to_int(packed_ip))
    except (AddrFormatError, struct.error):
        return


def pack_unsigned_int_be(value):
    return struct.pack('>I', value)


def unpack_unsigned_int_be(value):
    return struct.unpack('>I', value)[0]


def pack_long_be(value):
    packed = struct.pack('l', value)
    if sys.byteorder == 'little':
        return packed[::-1]
    return packed


def unpack_long_be(value):
    if sys.byteorder == 'little':
        value = value[::-1]
    return struct.unpack('l', value)[0]


def pack_unsigned_long_be(value):
    packed = struct.pack('L', value)
    if sys.byteorder == 'little':
        return packed[::-1]
    return packed


def unpack_unsigned_long_be(value):
    if sys.byteorder == 'little':
        value = value[::-1]
    return struct.unpack('L', value)[0]


def is_tor(ip):
    return get_geobase().is_tor(ip)


def geoid(ip):
    return Region(ip).id


def as_list(ip):
    return Region(ip).AS_list


def is_yandex_ip(ip):
    net = get_net(ip, {})
    return net['is_yandex'] and not net['is_user']


def none_or_true(value):
    return '1' if value else None


def none_or_lower(value):
    return value.lower() if value else None


def unpack_useragent(useragent):
    values = {}
    for k, v in get_uatraits().detect(useragent).items():
        v = parse_uatraits_value(v)
        if v is None:
            continue
        values[k] = v
    return values


def parse_uatraits_value(value):
    if not value or value == 'Unknown':
        return
    return value


def is_real_user_ip(ip):
    try:
        ip = IPAddress(ip)
    except AddrFormatError:
        return False

    # 127.0.0.1, ::1
    if ip.is_loopback():
        return False

    # 192.168.0.1, 10.0.0.1, etc
    if ip.is_private():
        return False

    if ip.is_multicast():
        return False
    return True


def entry_to_userip(entry):
    return entry.get('proxy_ip') or entry.get('user_ip')


def parse_comment(comment):
    if not comment:
        return {}
    items = {}
    for block in comment.split(';'):
        key, sep, value = block.partition('=')
        if not sep:
            continue
        items[key] = value
    return items
