# -*- coding: utf-8 -*-

import datetime
import logging
import os
import time

from passport.backend.utils.string import (
    smart_str,
    smart_unicode,
)
from passport.backend.utils.tskv import TskvLogEntry


REPLACE_MAP = [
    ('\\', r'\\'),
    ('\t', r'\t'),
    ('\n', r'\n'),
    ('\r', r'\r'),
    ('\0', r'\0'),
]


def escape_text_for_log(text, errors='strict'):
    text = smart_str(text, errors=errors)
    for char, replacement in REPLACE_MAP:
        text = text.replace(char, replacement)
    return text


class DefaultFormatter(logging.Formatter):
    def format(self, record):
        if hasattr(record, 'unixtime'):
            unixtime = record.unixtime
        else:
            unixtime = int(time.time())

        if hasattr(record, 'formatted_unixtime'):
            formatted_unixtime = record.formatted_unixtime
        else:
            formatted_unixtime = int(unixtime)

        if hasattr(record, 'request') and record.request:
            prefix = u'{user_ip} - - [{dt}] {unixtime} {request_id} "{method} {path}" '.format(
                user_ip=record.user_ip,
                dt=datetime.datetime.fromtimestamp(unixtime).strftime(self.datefmt),
                unixtime=formatted_unixtime,
                request_id=record.request_id,
                method=record.method,
                path=record.path,
            )
        else:
            prefix = u'{pid} {dt} {unixtime} '.format(
                pid=os.getpid(),
                dt=datetime.datetime.fromtimestamp(unixtime).strftime(self.datefmt),
                unixtime=formatted_unixtime,
            )
        message = smart_unicode(escape_text_for_log(record.getMessage()))
        return prefix + message


class StatboxFormatter(logging.Formatter):
    def __init__(self, fmt=None, datefmt=None, extra_schema=None):
        super(StatboxFormatter, self).__init__(fmt=fmt, datefmt=datefmt)
        self.extra_schema = extra_schema

    def format(self, record):
        message = dict(record.msg)
        message['unixtime'] = record.formatted_unixtime
        message['datetime'] = datetime.datetime.fromtimestamp(record.unixtime).strftime(self.datefmt)

        if record.request:
            message['request_id'] = record.request_id
            message['user_ip'] = record.user_ip
            message['user_agent'] = record.user_agent
            message['path'] = record.path
            message['method'] = record.method
            if hasattr(record, 'remote_ip'):
                message['remote_ip'] = record.remote_ip

        if self.extra_schema:
            for source in self.extra_schema:
                if source in message:
                    message[self.extra_schema[source]] = message[source]

        return str(TskvLogEntry(**message))
