# -*- coding: utf-8 -*-
import logging
import os
import sys

from jinja2 import Environment
from passport.backend.library.nginx_config_generator.filters import (
    format_ip,
    regex_escape,
)


log = logging.getLogger('nginx_config_generator')


def _make_template_name(config_dir, priority, template_name):
    return '%s/%02d_%s' % (config_dir, priority, template_name)


def _setup_and_get_env(template_dir=None, extra_filters=None):
    is_arcadia_python = getattr(sys, 'is_standalone_binary', False)
    if is_arcadia_python:
        from passport.backend.utils.template_loader import CachedTemplateLoader

        file_loader = CachedTemplateLoader(searchpath='')
    else:
        from jinja2 import FileSystemLoader

        assert template_dir is not None, 'You must specify `template_dir` for using this code outside Arcadia'
        cur_dir = os.path.abspath(os.path.dirname(__file__))
        file_loader = FileSystemLoader([cur_dir, template_dir])

    environment = Environment(
        loader=file_loader,
        keep_trailing_newline=True,
    )

    extra_filters = extra_filters or {}
    extra_filters.setdefault('format_ip', format_ip)
    extra_filters.setdefault('regex_escape', regex_escape)
    environment.filters.update(extra_filters)

    return environment


def _process_template(env, config, template_filename, destination_filename, create_empty_instead_of_missing=False):
    if config.is_applicable:
        template = env.get_template(template_filename)
        settings = config.build_settings()
        content = template.render(**settings)
    elif create_empty_instead_of_missing:
        log.debug('Creating empty config for %s', config.template_name)
        content = ''
    else:
        log.debug('Skipping config generation for %s', config.template_name)
        return

    with open(destination_filename, 'w') as f:
        f.write(content)


def build_configs(config_classes_with_priorities, config_dir, template_dir=None, extra_filters=None,
                  create_empty_instead_of_missing=False):
    log.info('Building nginx configs ...')
    env = _setup_and_get_env(template_dir=template_dir, extra_filters=extra_filters)

    for priority, config_cls in config_classes_with_priorities.items():
        config = config_cls()
        _process_template(
            env=env,
            config=config,
            template_filename='%s.j2' % config.template_name,
            destination_filename=_make_template_name(config_dir, priority, config.result_name),
            create_empty_instead_of_missing=create_empty_instead_of_missing,
        )

    log.info('Configs successfully built.')


def delete_configs(config_classes_with_priorities, config_dir):
    log.info('Deleting nginx configs ...')

    for priority, config_cls in config_classes_with_priorities.items():
        config = config_cls()
        filename = _make_template_name(config_dir, priority, config.result_name)
        try:
            os.remove(filename)
            log.debug('Deleted %s', filename)
        except OSError as e:
            log.debug('Unable to delete %s: %s', filename, e)

    log.info('Configs deleted.')


def enable_logging_to_stdout():
    log.addHandler(logging.StreamHandler())
    log.setLevel(logging.DEBUG)
