# -*- coding: utf-8 -*-
import logging
from multiprocessing import cpu_count
import socket

from passport.backend.library.nginx_config_generator.utils import (
    camel_case_to_snake_case,
    get_fqdn,
    get_ip,
)
import yenv


log = logging.getLogger('nginx_config_generator')


class BaseConfig(object):
    _cert_names = {}  # (yenv_name, yenv_type) -> str

    # servername может задаваться либо явным списком, либо через prefix + tlds
    _server_names = {}  # (yenv_name, yenv_type) -> list
    _server_name_prefixes = {}  # (yenv_name, yenv_type) -> list|str
    _tlds = {}  # (yenv_name, yenv_type) -> list

    _slb_ips = {}  # (yenv_name, yenv_type) -> list
    _localhost_ips = (
        '127.0.0.1',
        '::1',
    )
    _ssl_required = True

    def __init__(self):
        self.yenv_name = yenv.name
        self.yenv_type = yenv.type
        self.fqdn = get_fqdn()

    @property
    def is_applicable(self):
        """Нужно ли разворачивать данный конфиг в данном окружении"""
        # Если сертификата нет - значит, в этом окружении подниматься не надо
        if self._ssl_required and self._get_cert_name() is None:
            return False
        return True

    @property
    def template_name(self):
        return camel_case_to_snake_case(self.__class__.__name__)

    @property
    def result_name(self):
        return camel_case_to_snake_case(self.__class__.__name__)

    def _get_cert_name(self):
        """Возвращает имя SSL-сертификата (без пути и расширения)"""
        return self._cert_names.get(self.yenv_name, {}).get(self.yenv_type)

    def _get_server_name_prefixes(self):
        prefixes = self._server_name_prefixes.get(self.yenv_name, {}).get(self.yenv_type, [])
        if not isinstance(prefixes, (list, tuple)):
            prefixes = [prefixes]
        return prefixes

    def _get_tlds(self):
        """Возвращает имена/регулярки хостов, на которых слушаем"""
        return self._tlds.get(self.yenv_name, {}).get(self.yenv_type, [])

    def _get_server_names(self):
        """Возвращает имена/регулярки хостов, на которых слушаем"""
        server_names = self._server_names.get(self.yenv_name, {}).get(self.yenv_type)
        prefixes, tlds = self._get_server_name_prefixes(), self._get_tlds()
        if server_names and (prefixes or tlds):
            raise ValueError(
                'Config %s is ambiguous: either servernames or (server_name_prefixes + tlds) must be set, but not both' % (
                    self.__class__.__name__,
                ),
            )

        if server_names:
            return server_names
        elif prefixes and tlds:
            return [
                '.'.join([prefix, tld])
                for tld in tlds
                for prefix in prefixes
            ]

    def _get_server_ips(self):
        """Возвращает IP-адреса текущей машины"""
        server_ips = []
        # Пытаемся получить IPv4 и IPv6 адреса
        for ip_version, ip_family in (
            (4, socket.AF_INET),
            (6, socket.AF_INET6),
        ):
            try:
                server_ips.append(get_ip(self.fqdn, ip_family))
            except socket.error:
                # Например, в Финке нет IPv4 - getaddrinfo выкинет ошибку для socket.AF_INET
                log.warning('Failed to discover server\'s IPv%d address', ip_version)
                continue
        return tuple(server_ips) + tuple(self._localhost_ips)

    def _get_slb_ips(self):
        """Возвращает IP-адреса фермы"""
        return self._slb_ips.get(self.yenv_name, {}).get(self.yenv_type, [])

    def build_settings(self):
        """Строит словарь настроек, необходимых для рендера шаблона"""
        if not self.is_applicable:
            raise ValueError(
                'Cannot build %s on %s (%s %s)' % (
                    self.__class__.__name__,
                    self.fqdn,
                    self.yenv_name,
                    self.yenv_type,
                ),
            )  # pragma: no cover

        return self.get_custom_settings()

    def get_custom_settings(self):
        return {
            'cert_name': self._get_cert_name(),
            'server_names': self._get_server_names(),
            'server_name_prefixes': self._get_server_name_prefixes(),
            'tlds': self._get_tlds(),
            'slb_ips': self._get_slb_ips(),
            'server_ips': self._get_server_ips(),
            'cpu_count': cpu_count(),
            'yenv': yenv,
        }
