# -*- coding: utf-8 -*-
import datetime
import getpass
import os
import re

import debian.changelog
from passport.backend.library.packaging.exceptions import CantFindProjectDirectoryError
from passport.backend.utils.file import mkdir_p
import semver
from six.moves import input


def format_commit(message, package_name, omit_lines=None):
    # удаляем лишние подстроки
    for regex in (
        r'\[mergeto:[\w-]+:\d+\]',
    ):
        message = re.sub(regex, '', message)

    # удаляем лишние строки
    message = [ll.rstrip() for ll in message.splitlines() if ll.strip()]
    message = filter(
        lambda x: (
            not x.startswith('Pull-request for branch') and
            not x.startswith('REVIEW:') and
            not ('release' in x and package_name in x) and
            not x.startswith('Note: ') and
            '[arc::pullid]' not in x and
            'BYPASS_CHECKS' not in x and
            '<!-- DEVEXP BEGIN -->' not in x and
            '<!-- DEVEXP END -->' not in x and
            not x.startswith('![review]') and
            (not omit_lines or not any([re.match(regex, x) for regex in omit_lines]))
        ),
        message,
    )

    # удаляем дублирующиеся строки
    unique_filter = set()
    unique_message_filter = []
    for line in message:
        if line in unique_filter:
            continue
        unique_filter.add(line)
        unique_message_filter.append(line)

    # собираем коммит-мессадж обратно
    message = '\n'.join(unique_message_filter)
    message = message.replace('\n', '\n    ').strip()
    return message


def format_commits(entries, package_name, omit_names=False, omit_lines=None):
    last_author = None
    lines = []
    for entry in entries:
        if not omit_names and (not last_author or last_author != entry.author):
            last_author = entry.author
            lines.append('  [ %s ]' % last_author)
        new_part = format_commit(entry.message, package_name, omit_lines=omit_lines)
        if new_part:
            lines.append('  * %s' % new_part)
    return lines


def reverse_format_commits(lines):
    entries = []
    for line in lines:
        line = line.strip()
        if not line or re.match(r'\[\s[\w]+\s\]', line):
            continue
        line = line.lstrip('*')
        line = line.strip()
        entries.append(line)
    return entries


class Packaging(object):
    def __init__(self, base_path='.', deb_path='deb'):
        if base_path is None:
            base_path = '.'
        if deb_path is None:
            deb_path = 'deb'

        self.base_path = base_path
        self.deb_path = deb_path

        current_dir_name = os.path.basename(os.getcwd())

        if os.path.exists(os.path.join(self.base_path, self.deb_path)):
            self.changelog_path = os.path.join(self.base_path, self.deb_path, 'debian/changelog')
        elif current_dir_name == deb_path:
            self.changelog_path = 'debian/changelog'
            self.base_path = '../'
            self.deb_path = './'
        else:
            raise CantFindProjectDirectoryError(
                'base_path="%s". deb_path="%s". current_dir_name="%s"' % (
                    self.base_path,
                    self.deb_path,
                    current_dir_name,
                )
            )

    def ensure(self):
        if not os.path.exists(os.path.dirname(self.changelog_path)):
            mkdir_p(os.path.dirname(self.changelog_path))

        if not os.path.exists(self.changelog_path):
            open(self.changelog_path, 'w').close()

        if os.stat(self.changelog_path).st_size != 0:
            with open(self.changelog_path) as changelog_file:
                changelog = debian.changelog.Changelog(changelog_file)

            package = changelog.get_package()
            version = changelog.get_version()
            return changelog, package, str(version)
        else:
            return None, None, None

    def increase_version(self, entries, new_version=None, omit_names=False, omit_lines=None, version_increase=None, always_yes=False):
        changelog, package, version = self.ensure()

        if not changelog:
            package = input('Enter package name: ')
            new_version = new_version or '0.0.1'
            with open(self.changelog_path) as changelog_file:
                changelog = debian.changelog.Changelog(changelog_file)

        if not new_version:
            if not version_increase:
                version_increase = 'patch'
            increaser = getattr(semver, 'bump_%s' % version_increase)
            new_version = increaser(version)

        commits = format_commits(entries, package_name=package, omit_names=omit_names, omit_lines=omit_lines)
        if not commits:
            commits = ['  * Dummy release']

        changelog.new_block(
            package=package,
            version=new_version,
            distributions='unstable',
            author='%s <%s>' % (os.getenv('DEBFULLNAME') or getpass.getuser(), os.getenv('DEBEMAIL') or getpass.getuser() + '@yandex-team.ru'),
            date=datetime.datetime.now().strftime("%a, %d %b %Y %H:%M:%S +0300"),
            changes=[''] + commits + [''],
            urgency='low',
        )
        if always_yes:
            print('Adding new changelog entry:')
        else:
            print('Confirm addition the new changelog entry:')

        print(str(changelog._blocks[0]))

        if not (always_yes or input('Are you sure? (y/n) ') == 'y'):
            return changelog, None, None

        with open(self.changelog_path, 'w') as changelog_file:
            changelog.write_to_open_file(changelog_file)

        return changelog, package, new_version

    def get_last_block_commits(self):
        changelog, _, _ = self.ensure()
        return reverse_format_commits(changelog._blocks[0].changes())

    def get_last_changelog(self):
        changelog, _, _ = self.ensure()
        return '\n'.join(changelog._blocks[0].changes())
