package passportapi

import (
	"context"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"
)

func headersToReq(headers Headers) map[string]string {
	retval := map[string]string{}
	if len(headers.YaClientIP) > 0 {
		retval["Ya-Consumer-Client-IP"] = headers.YaClientIP
	}
	if len(headers.YaClientHost) > 0 {
		retval["Ya-Client-Host"] = headers.YaClientHost
	}
	if len(headers.YaClientCookie) > 0 {
		retval["Ya-Client-Cookie"] = headers.YaClientCookie
	}
	if len(headers.YaClientUserAgent) > 0 {
		retval["Ya-Client-User-Agent"] = headers.YaClientUserAgent
	}
	if len(headers.YaClientXRequestID) > 0 {
		retval["Ya-Client-X-Request-ID"] = headers.YaClientXRequestID
	}
	return retval
}

func (c *client) request(ctx context.Context, method string, path string, headers Headers, formData url.Values) (int, []byte, error) {
	var (
		statusCode int
		body       []byte
	)
	queryURL := c.Environment.BaseURL + path

	request, err := http.NewRequestWithContext(ctx, method, queryURL, nil)
	if err != nil {
		return statusCode, body, err
	}

	tvmTicket, err := c.TvmClient.GetServiceTicketForID(ctx, c.Environment.TVMID)
	if err != nil {
		return statusCode, body, err
	}

	query := request.URL.Query()
	query.Add("consumer", c.Consumer)
	request.URL.RawQuery = query.Encode()

	h := headersToReq(headers)
	for k, v := range h {
		request.Header.Add(k, v)
	}

	if len(formData) > 0 {
		body := io.NopCloser(strings.NewReader(formData.Encode()))
		request.Body = body
	}

	request.Header.Add("X-Ya-Service-Ticket", tvmTicket)
	request.Header.Add("Content-Type", "application/x-www-form-urlencoded")

	response, err := c.Client.Do(request)
	if err != nil {
		return statusCode, body, err
	}
	if response != nil {
		defer response.Body.Close()
	}
	statusCode = response.StatusCode
	body, err = ioutil.ReadAll(response.Body)
	if err != nil {
		return statusCode, body, err
	}
	return statusCode, body, nil
}
