import getpass
import json
import re

from dateutil.parser import parse
from passport.backend.library.repo.base import (
    BaseManager,
    Commit,
    CommitSearchResult,
)
from passport.backend.utils.regex import url_regex
from passport.backend.utils.system import run_command
from passport.backend.utils.time import datetime_to_integer_unixtime


class ArcManager(BaseManager):
    def _from_loaded(self, loaded_commit):
        return Commit(
            author=loaded_commit['author'],
            message=loaded_commit['message'],
            unixtime=datetime_to_integer_unixtime(parse(loaded_commit['date'])),
            hash=loaded_commit['commit'],
        )

    def pull_request(self, files, message, branch_name):
        run_command('arc checkout -b %s > /dev/null' % branch_name)
        run_command('arc add %s' % ' '.join(files))
        run_command('arc commit -m "%s"' % message)
        run_command('arc push -u users/{user_name}/{branch_name}'.format(
            user_name=getpass.getuser(),
            branch_name=branch_name,
        ))
        response = run_command('arc pr create --publish --auto --message="%s"' % message)
        run_command('arc checkout trunk > /dev/null')
        run_command('arc branch -D {branch_name}'.format(branch_name=branch_name))

        urls = re.findall(url_regex, response)
        if urls:
            return urls[0]

    def get_commits_since(self, directory='.', commit_hash=None, message=None, target_date=None, commit_limit=1024):
        target_unixtime = 0 if target_date is None else datetime_to_integer_unixtime(target_date)
        commits = []
        output = run_command(
            'arc log {directory} --json -n {commit_limit}'.format(
                directory=directory,
                commit_limit=commit_limit,
            ),
        )
        if not output:
            return commits
        loaded_commits = json.loads(output)
        for loaded_commit in loaded_commits:
            if isinstance(message, (tuple, list)):
                has_messages = True
                for el in message:
                    has_messages = has_messages and (el in loaded_commit['message'])
            elif isinstance(message, (str, bytes)):
                has_messages = message in loaded_commit['message']
            else:
                has_messages = False
            if (
                has_messages or
                loaded_commit['commit'] == commit_hash or
                target_unixtime > datetime_to_integer_unixtime(parse(loaded_commit['date']))
            ):
                return CommitSearchResult(
                    commits_since=commits,
                    found_commit=self._from_loaded(loaded_commit),
                )
            commits.append(self._from_loaded(loaded_commit))
        return CommitSearchResult(commits_since=commits, found_commit=None)

    def revert(self, file):
        return run_command('arc checkout %s' % file)

    def is_touched(self, file):
        return len(run_command('arc diff %s' % file)) > 0
