from datetime import (
    date,
    timedelta,
)
import getpass
import json
import re
from tempfile import NamedTemporaryFile

from passport.backend.library.repo.base import (
    BaseManager,
    Commit,
    CommitSearchResult,
)
from passport.backend.utils.regex import url_regex
from passport.backend.utils.system import run_command
from passport.backend.utils.time import (
    datetime_to_integer_unixtime,
    to_date_str,
)


class HgManager(BaseManager):
    def _from_loaded(self, loaded_commit):
        return Commit(
            author=re.search(r'^[\w\s]+<(\w+)@*.+>$', loaded_commit['user']).group(1),
            message=loaded_commit['desc'],
            unixtime=loaded_commit['date'][0],
            hash=loaded_commit['node'],
        )

    def pull_request(self, files, message, branch_name):
        branch_name = "users/{user_name}/{branch_name}".format(
            user_name=getpass.getuser(),
            branch_name=branch_name,
        )
        files = ' '.join(files)
        with NamedTemporaryFile() as tf_diff:
            run_command('ya tool hg diff {files} > {filename}'.format(files=files, filename=tf_diff.name))
            run_command('ya tool hg update default -C')
            run_command('ya tool hg branch {branch_name} -f'.format(branch_name=branch_name))
            run_command('ya tool hg import {filename} --no-commit'.format(filename=tf_diff.name))

        run_command('ya tool hg add {files}'.format(files=files))
        run_command('ya tool hg commit {files} -m "{message}"'.format(message=message, files=files))
        run_command('ya tool hg push -b . --new-branch')
        response = run_command('ya tool hg pull-request --open --require REVIEW --description "%s"' % message)
        run_command('ya tool hg update default')
        urls = re.findall(url_regex, response)
        if urls:
            return urls[0]

    def get_commits_since(self, directory='.', commit_hash=None, message=None, target_date=None, commit_limit=24):
        target_unixtime = 0 if target_date is None else datetime_to_integer_unixtime(target_date)
        commits = []
        output = run_command(
            'ya tool hg log {directory} --no-merges -Tjson -l {commit_limit} -M '
            '-d "{oldest_commit_date} to {today_date}"'.format(
                directory=directory,
                commit_limit=commit_limit,
                oldest_commit_date=to_date_str(date.today() - timedelta(days=90)),
                today_date=to_date_str(date.today()),
            ),
        )
        if not output:
            return commits
        loaded_commits = json.loads(output)
        for loaded_commit in loaded_commits:
            if isinstance(message, (tuple, list)):
                has_messages = True
                for el in message:
                    has_messages = has_messages and (el in loaded_commit['desc'])
            elif isinstance(message, (str, bytes)):
                has_messages = message in loaded_commit['desc']
            else:
                has_messages = False
            if (
                has_messages or
                loaded_commit['rev'] == commit_hash or
                target_unixtime > loaded_commit['date'][0]
            ):
                return CommitSearchResult(
                    commits_since=commits,
                    found_commit=self._from_loaded(loaded_commit),
                )
            commits.append(self._from_loaded(loaded_commit))

        return CommitSearchResult(commits_since=commits, found_commit=None)

    def revert(self, file):
        run_command('ya tool hg revert %s' % file)

    def is_touched(self, file):
        return len(run_command('ya tool hg diff %s' % file)) > 0
