import re

from dateutil.parser import parse
from passport.backend.library.repo.base import (
    BaseManager,
    Commit,
    CommitSearchResult,
)
from passport.backend.utils.regex import url_regex
from passport.backend.utils.system import run_command
from passport.backend.utils.time import datetime_to_integer_unixtime
import xmltodict


def _ignored_pull_request_url(url):
    return any(e.match(url) for e in _ignored_pull_request_url.re_list)


_ignored_pull_request_url.re_list = [re.compile(e) for e in [
    r'^https://proxy\.sandbox\.yandex-team\.ru/',
]]


class SvnManager(BaseManager):
    def _get_revision_with_message(self, directory, message):
        if isinstance(message, (tuple, list)):
            condition = '--search "%s"' % message[0]
            for el in message[1:]:
                condition += ' --search-and "%s"' % el
        else:
            condition = '--search "%s"' % message
        output = run_command('ya tool svn log --xml {condition} {directory} '.format(
            condition=condition,
            directory=directory,
        ))
        log = xmltodict.parse(output)['log']
        if log and 'logentry' in log:
            if isinstance(log['logentry'], (tuple, list)):
                return int(log['logentry'][0]['@revision'])
            return int(log['logentry']['@revision'])
        return 0

    def pull_request(self, files, message, branch_name):
        response = run_command(
            "ya pr create --no-interactive --publish --auto -m '{message}' {files} 2>&1".format(
                message=message,
                files=' '.join(files),
            ),
        )
        urls = re.findall(url_regex, response)
        for url in urls:
            if not _ignored_pull_request_url(url):
                return url

    def get_commits_since(self, directory='.', commit_hash=None, message=None, target_date=None):
        target_unixtime = 0 if target_date is None else datetime_to_integer_unixtime(target_date)

        if not commit_hash and message:
            commit_hash = self._get_revision_with_message(directory=directory, message=message)

        if commit_hash:
            output = run_command('ya tool svn log --xml -r {revision}:HEAD {directory}'.format(
                revision=commit_hash,
                directory=directory
            ))
        else:
            output = run_command('ya tool svn log --xml {directory}'.format(
                directory=directory
            ))

        commits = (xmltodict.parse(output).get('log') or {}).get('logentry', [])
        if not isinstance(commits, (tuple, list)):
            commits = [commits]

        commits = list(map(
            lambda x: Commit(
                author=x['author'],
                message=x['msg'],
                unixtime=datetime_to_integer_unixtime(parse(x['date'])),
                hash=x['@revision'],
            ),
            commits
        ))
        if target_unixtime:
            filtered_commits = []
            for commit in commits:
                if target_unixtime < commit.unixtime:
                    filtered_commits.append(commit)
            commits = filtered_commits
        if commits:
            return CommitSearchResult(commits_since=commits[1:], found_commit=commits[0])
        else:
            return CommitSearchResult(commits_since=[], found_commit=None)

    def revert(self, file):
        return len(run_command('ya tool svn revert %s' % file)) > 0

    def is_touched(self, file):
        return len(run_command('ya tool svn diff %s' % file)) > 0
