# coding: utf-8
import logging
import resource

from gunicorn.app.base import BaseApplication
from passport.backend.library.wsgi_runner.exceptions import InvalidConfigurationError
from passport.backend.utils.common import noneless_dict
from passport.backend.utils.importlib import import_object
from werkzeug.serving import run_simple


log = logging.getLogger('passport')


class StandaloneApplication(BaseApplication):
    def __init__(self, app, options=None):
        self.options = options or {}
        self.application = app
        super(StandaloneApplication, self).__init__()

    def load_config(self):
        config = dict([
            (key, value) for key, value in self.options.items()
            if key in self.cfg.settings and value is not None
        ])
        for key, value in config.items():
            self.cfg.set(key.lower(), value)

    def load(self):
        return self.application


class Runner(object):
    def __init__(self, app, config):
        self.app = app
        self.config = config
        if 'application' not in self.config:
            raise InvalidConfigurationError('No `application` section in config')

    def _process_directives(self):
        directives = self.config['application'].get('directives', {})
        if directives.get('nofile') is not None:
            resource.setrlimit(resource.RLIMIT_NOFILE, (directives['nofile'], directives['nofile']))

    def run_gunicorn(self, skip_monkey_patch=False):
        application = self.config['application']
        gunicorn = self.config.get('gunicorn', {})

        bind_to = application.get('bind', [])
        if application.get('port'):
            bind_to.append('%s:%s' % (
                application.get('address', '127.0.0.1'),
                application['port'],
            ))
        if application.get('unix_socket_file'):
            bind_to.append('unix:%s' % application['unix_socket_file'])
        if not bind_to:
            raise InvalidConfigurationError('At least one of (`bind`, `port`, `unix_socket_file`) must be specified')

        options = {
            'bind': bind_to,
            'capture_output': gunicorn.get('capture_output', True),
            'errorlog': gunicorn.get('log_file', 'gunicorn.log'),
            'workers': application.get('workers', 1),
            'worker_class': gunicorn.get('worker_class', 'sync'),
        }
        options.update(noneless_dict({
            'accesslog': gunicorn.get('access_log_file'),
            'keepalive': gunicorn.get('keepalive'),
            'pidfile': gunicorn.get('pidfile'),
            'post_worker_init': application.get('post_worker_init'),
            'preload': gunicorn.get('preload', False),
            'reuse_port': gunicorn.get('reuse_port', True),
            'timeout': gunicorn.get('timeout'),
            'worker_connections': gunicorn.get('worker_connections'),
            'worker_tmp_dir': gunicorn.get('worker_tmp_dir'),
        }))

        if options['worker_class'] == 'gevent':
            if not skip_monkey_patch:
                from gevent import monkey
                monkey.patch_all()

        log.debug('Running gunicorn server on {} with {}'.format(
            ', '.join(addr for addr in bind_to),
            options,
        ))
        return StandaloneApplication(self.app, options).run()

    def run_native(self):
        application = self.config['application']
        options = {
            'hostname': application.get('address', '127.0.0.1'),
            'port': int(application.get('port', '9000')),
            'application': self.app,
            'use_debugger': True,
        }
        post_worker_init = application.get('post_worker_init')
        if post_worker_init:
            import_object(post_worker_init)(None)
        log.debug('Running native server with {}'.format(options))
        return run_simple(**options)

    def run(self):
        application = self.config['application']
        self._process_directives()
        getattr(self, 'run_%s' % application.get('server_type', 'native'))()
