import numpy as np
from sklearn.metrics import (
    confusion_matrix,
    log_loss,
    roc_auc_score,
)


precision = 0.000001


class ClassificationReport(object):
    def __init__(self, y_test, y_pred):
        self.y_test = np.asarray(y_test)
        self.y_pred = np.array(y_pred)
        self.y_pred[np.where(np.isnan(self.y_pred) | np.isinf(self.y_pred))] = 0.5
        self.y_pred[np.where(self.y_pred < precision)] = precision
        self.y_pred[np.where(self.y_pred > 1 - precision)] = 1 - precision

    def __iadd__(self, other):
        self.y_test = np.append(self.y_test, other.y_test)
        self.y_pred = np.append(self.y_pred, other.y_pred)
        return self

    def log_loss(self):
        return log_loss(self.y_test, self.y_pred)

    def auc(self):
        return roc_auc_score(self.y_test, self.y_pred)

    def p_support(self):
        return self.y_test.sum() / float(self.y_test.shape[0])

    def n_support(self):
        return 1 - (self.y_test.sum() / float(self.y_test.shape[0]))

    def threshold_statistics(self, thresholds):
        rows = []
        for thre in thresholds:
            row = {}
            cm = confusion_matrix(
                self.y_test,
                (self.y_pred > thre).astype(int),
                labels=[0, 1],
            )
            P = self.y_test.sum()
            N = len(self.y_test) - P
            row['threshold'] = thre
            row['tnr'] = cm[0][0] / float(N)
            row['tpr'] = cm[1][1] / float(P)
            row['fpr'] = cm[0][1] / float(N)
            row['fnr'] = cm[1][0] / float(P)
            row['tc'] = (self.y_pred > thre).sum() / float(self.y_test.shape[0])
            row['fc'] = (self.y_pred < thre).sum() / float(self.y_test.shape[0])
            rows.append(row)
        return rows
