#include "interval_counters.h"

template<class key_type, class val_type>
void IntervalCounters<key_type, val_type>::add(const key_type& key, const val_type& val){
    counters[key].push_back(val);
}

template<class key_type, class val_type>
uint32_t IntervalCounters<key_type, val_type>::count(const key_type& key, const val_type& begin, const val_type& end){
    const auto& counter = counters.at(key);
    const auto& left_it = lower_bound(counter.begin(), counter.end(), begin);
    const auto& right_it = lower_bound(counter.begin(), counter.end(), end);
    return right_it - left_it;
}

template<class key_type, class val_type>
uint32_t IntervalCounters<key_type, val_type>::count_before(const key_type& key, const val_type& end){
    const auto& counter = counters.at(key);
    const auto& it = lower_bound(counter.begin(), counter.end(), end);
    return it - counter.begin();
}

template<class key_type, class val_type>
uint32_t IntervalCounters<key_type, val_type>::count_after(const key_type& key, const val_type& begin){
    const auto& counter = counters.at(key);
    const auto& it = upper_bound(counter.begin(), counter.end(), begin);
    return counter.end() - it;
}

template<class key_type, class val_type>
bool IntervalCounters<key_type, val_type>::has(const key_type& key){
    return (counters.find(key) != counters.end());
}

// Partial Interval Counters

template<class key_type, class val_type>
void PartialIntervalCounters<key_type, val_type>::add(const key_type& key, const val_type& val, uint32_t counter){
    const auto& counter_vector = PartialIntervalCounters::counters[key];
    if (counter_vector.size() > 0) {
        counter += counter_vector.back().second;
    }
    counters[key].push_back({val, counter});
}

template<class key_type, class val_type>
uint32_t PartialIntervalCounters<key_type, val_type>::count(const key_type& key, const val_type& begin, const val_type& end){
    const auto& counter = counters.at(key);
    const auto& left_it = upper_bound(counter.rbegin(), counter.rend(), begin, this->GreaterComparator);
    const auto& right_it = upper_bound(counter.rbegin(), counter.rend(), end, this->GreaterComparator);
    if (left_it != counter.rend())
        return (right_it->second - left_it->second);
    else if (right_it != counter.rend())
        return right_it->second;
    else
        return 0;
}

template<class key_type, class val_type>
uint32_t PartialIntervalCounters<key_type, val_type>::count_before(const key_type& key, const val_type& end){
    const auto& counter = counters.at(key);
    const auto& it = upper_bound(counter.rbegin(), counter.rend(), end, this->GreaterComparator);
    if (it != counter.rend())
        return it->second;
    else
        return 0;
}

template<class key_type, class val_type>
uint32_t PartialIntervalCounters<key_type, val_type>::count_after(const key_type& key, const val_type& begin){
    const auto& counter = counters.at(key);
    const auto& it = lower_bound(counter.rbegin(), counter.rend(), begin, this->GreaterComparator);
    if (it != counter.rend())
        return counter.rbegin()->second - it->second;
    else
        return counter.rbegin()->second;
}

template<class key_type, class val_type>
bool PartialIntervalCounters<key_type, val_type>::has(const key_type& key){
    return (counters.find(key) != counters.end());
}

template class PartialIntervalCounters<uint64_t, uint32_t>;
template class IntervalCounters<uint64_t, uint32_t>;
template class IntervalCounters<uint32_t, uint32_t>;
template class IntervalCounters<std::string, uint32_t>;
