# -*- coding: utf-8 -*-
import logging
import os
import sys
import unittest

from passport.backend.libs_checker import library_tests
from passport.backend.libs_checker.environment import StagingEnvironment
from passport.backend.libs_checker.exceptions import (
    BaseLibsCheckerError,
    SourceNotFoundError,
    TargetNotAFileError,
    TestsFailedError,
)
from passport.backend.libs_checker.utils import make_target_path


log = logging.getLogger('passport_libs_checker.info')


LIBS_TO_CHECK = {
    'libgeobase': library_tests.geobase_checker.TestGeobase,
    'libipreg': library_tests.ipreg_checker.TestLibIpreg,
    'uatraits': library_tests.uatraits_checker.TestUATraits,
}


FLOAT_INSIGNIFICANT_DELTA = 1e-3


def get_updated_and_missing_libs(config, target_libs):
    updated_libs = set()
    missing_libs = set()

    for library_name in target_libs:
        source_paths = config['supported_libs'][library_name]
        if library_name in missing_libs:
            continue
        for source_path in source_paths.values():
            if not os.path.isfile(source_path):
                log.error('Source file not found: "%s"', source_path)
                missing_libs.add(library_name)
                break

            target_path = make_target_path(source_path)
            if os.path.exists(target_path) and not os.path.isfile(target_path):
                log.error('"%s": file expected', target_path)
                raise TargetNotAFileError()

            source_modification_time = os.path.getmtime(source_path)
            target_modification_time = os.path.getmtime(target_path) if os.path.exists(target_path) else 0

            if target_modification_time - source_modification_time > FLOAT_INSIGNIFICANT_DELTA:
                log.error('Target file "%s" is newer than source file "%s"', target_path, source_path)

            if source_modification_time - target_modification_time > FLOAT_INSIGNIFICANT_DELTA:
                updated_libs.add(library_name)

    return updated_libs, missing_libs


def run_tests(libs_to_check, verbose=False):
    verbosity = 2 if verbose else 1

    results = []

    for library_name in libs_to_check:
        test = LIBS_TO_CHECK[library_name]
        suite = unittest.TestSuite()
        for method in dir(test):
            if method.startswith('test_'):
                suite.addTest(test(method))
        runner = unittest.TextTestRunner(
            verbosity=verbosity,
            stream=sys.stderr if verbose else open(os.devnull, 'w'),
        )
        result = runner.run(suite)
        results.append(result.wasSuccessful())
    return all(results)


def libs_checker(config, target_libs, verbose=False):
    log.info('Started processing libs: %s', ', '.join(target_libs))
    try:
        updated_libs, missing_libs = get_updated_and_missing_libs(config, target_libs)
        if not updated_libs:
            log.info('No updates found')
        else:
            log.info('Checking updated libs: %s', ', '.join(updated_libs))
            libs_to_check = set(target_libs) - missing_libs
            log.info('Running tests in original environment')
            StagingEnvironment.setup_original_environment(libs_to_check)
            # На первом шаге проверяем исходные версии файлов, чтобы не делать лишних копирований
            tests_passed = False
            if run_tests(libs_to_check, verbose):
                log.info('Copying libs to target path and running tests again')
                StagingEnvironment.setup_target_environment(libs_to_check, updated_libs)
                # На втором шаге делаем прогон тестов в изолированном окружении
                if run_tests(libs_to_check, verbose):
                    StagingEnvironment.commit_changes()
                    tests_passed = True

            if not tests_passed:
                log.error('Tests failed')
                raise TestsFailedError()

            log.info('Done')

        if missing_libs:
            raise SourceNotFoundError()
    except BaseLibsCheckerError:
        raise
    except Exception as e:
        log.exception('Unhandled exception: %s: %s', e.__class__.__name__, e)
        raise
