# -*- coding: utf-8 -*-
import inspect
import sys

from passport.backend.library.historydbloader.historydb.parser import ParserType
from passport.backend.logbroker_client.core.events.events import (
    AccountFieldSetToValueEvent,
    AccountUnsubscribeEvent,
    UserEvent,
)


def get_module_classes_by_name(name):
    classes_pairs = inspect.getmembers(
        object=sys.modules[__name__],
        predicate=lambda cls: inspect.isclass(cls) and cls.__name__.startswith(name)
    )
    return map(lambda x: x[1], classes_pairs)


class AccountUnsubscribeWithSidEvent(AccountUnsubscribeEvent):
    EXPECTED_SIDS = set()

    @classmethod
    def is_this_event(cls, parser_type, event, expected_sids=None):
        event = AccountUnsubscribeEvent.is_this_event(parser_type, event, cls.EXPECTED_SIDS)
        if event is not None:
            return cls(event.uid, event.timestamp, event.sid)


class AccountUnsubscribeDiskEvent(AccountUnsubscribeWithSidEvent):
    EXPECTED_SIDS = [59]


class AccountPlusEvent(UserEvent):
    """
    Событие включения/выключения плюса
    """

    NAME = 'account.plus'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'plus.enabled':
            return cls(event.get('uid'), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AccountPlusEvent, self).__init__(self.NAME, uid, timestamp)


class AccountOtpEnabledEvent(AccountFieldSetToValueEvent):
    """
    Включение OTP
    """

    NAME = 'account.otp_enabled'
    FIELD_NAME = 'info.totp'
    FIELD_VALUE = 'enabled'


class AccountOtpDisabledEvent(AccountFieldSetToValueEvent):
    """
    Выключение OTP
    """

    NAME = 'account.otp_disabled'
    FIELD_NAME = 'info.totp'
    FIELD_VALUE = 'disabled'


class AccountDisableEvent(UserEvent):
    # событие блоикировки пользователя
    NAME = 'account.disable'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('name') == 'info.disabled_status' and
                event.get('value') == '1'):
            return cls(event.get('uid'), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AccountDisableEvent, self).__init__(self.NAME, uid, timestamp)


class AccountEnableEvent(UserEvent):
    NAME = 'account.enable'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('name') == 'info.disabled_status' and
                event.get('value') == '0'):
            return cls(event.get('uid'), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AccountEnableEvent, self).__init__(self.NAME, uid, timestamp)


class AccountRemovalDisableEvent(UserEvent):
    NAME = 'account.removal_disable'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('name') == 'info.disabled_status' and
                event.get('value') == '2'):
            return cls(event.get('uid'), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AccountRemovalDisableEvent, self).__init__(self.NAME, uid, timestamp)


EVENTS_WITHOUT_CHECKING_SERVICE = (
    get_module_classes_by_name('AccountUnsubscribe') +
    get_module_classes_by_name('AccountPlusEvent')
)
