# -*- coding: utf-8 -*-
import logging
import time

from passport.backend.core.builders.blackbox.blackbox import Blackbox
from passport.backend.core.builders.blackbox.exceptions import BlackboxTemporaryError
from passport.backend.core.builders.datasync_api import DiskApi
from passport.backend.core.builders.edadeal import EdadealApi
from passport.backend.core.tvm import TvmCredentialsManager
from passport.backend.logbroker_client.account_events.events import (
    AccountFieldSetToValueEvent,
    EVENTS_WITHOUT_CHECKING_SERVICE,
)
from passport.backend.logbroker_client.account_events.exceptions import BaseNotifyError
from passport.backend.logbroker_client.account_events.filter import ServiceFilter
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.core.utils import (
    ExponentialBackoff,
    importobj,
    is_yateam_uid,
)


log = logging.getLogger(__name__)


class ServiceHandlerException(HandlerException):
    pass


class EventsHandler(BaseHandler):
    handler_name = 'account_events'
    EVENT_NAME = 'account.changed'

    def __init__(self, config, dry_run, blackbox_url, disk_url, edadeal_url, edadeal_token, services,
                 **kwargs):
        super(EventsHandler, self).__init__(config=config, **kwargs)
        tvm_credentials_manager = TvmCredentialsManager(
            keyring_config_name='logbroker-client',
        )
        tvm_credentials_manager.load()

        self.dry_run = dry_run
        self.blackbox = Blackbox(blackbox_url, use_tvm=False)
        self.disk = DiskApi(
            disk_url,
            timeout=10.0,
            retries=3,
            tvm_credentials_manager=tvm_credentials_manager,
        )
        self.edadeal = EdadealApi(
            edadeal_url,
            timeout=1,
            retries=2,
            token=edadeal_token,
        )
        self.config = {}
        self.sids = []
        # FIXME: backoff общий на все сервисы, что не всегда корректно
        self.backoff = ExponentialBackoff(init_value=1, max_value=300)
        self.failed_services = set()
        all_events_classes = set()

        for service_name, service_config in services.items():
            notify_client_class = importobj(service_config['notifier'])
            events_classes = [importobj(event) for event in service_config['events']]
            all_events_classes = all_events_classes.union(set([event for event in service_config['events']]))

            self.config.setdefault(service_name, {}).update(
                client=notify_client_class(
                    service_name,
                    service_config.get('url'),
                    blackbox=self.blackbox,
                    disk=self.disk,
                    edadeal=self.edadeal,
                ),
            )
            sid = service_config.get('sid')
            self.config[service_name]['filter'] = ServiceFilter(events_classes, sid)
            if sid:
                self.sids.append(sid)
        self.uniting_filter = BasicFilter([importobj(event) for event in list(all_events_classes)])

    def get_sids_for_event(self, event):
        dbfields = ['subscription.suid.%s' % sid for sid in self.sids]
        userinfo = self.blackbox.userinfo(
            uid=event.uid,
            dbfields=dbfields,
            attributes=[],
            need_display_name=False,
            need_public_name=False,
            need_aliases=False,
        )
        if userinfo.get('uid') is None:
            sids = []
        else:
            sids = userinfo.get('subscriptions', {}).keys()
        return sids

    def get_event_name_for_event(self, event):
        if isinstance(event, AccountFieldSetToValueEvent):
            return event.name
        return self.EVENT_NAME

    def parse_message(self, message):
        return self.uniting_filter.filter(message)

    def process(self, header, data):
        message = MessageChunk(header, data)
        union_events = self.get_message_entries(message)
        events_sids = {}

        # получаем подписки для пользователя, которому принадлежит событие из ЧЯ
        for event in union_events:
            if event.__class__ not in EVENTS_WITHOUT_CHECKING_SERVICE:
                try:
                    events_sids[event] = self.get_sids_for_event(event)
                except BlackboxTemporaryError as e:
                    raise ServiceHandlerException(e)

        for service, service_config in self.config.items():
            client = service_config['client']
            events = service_config['filter'].filter(union_events, events_sids)
            for event in events:
                self.process_event(client, event)
        return True

    def process_event(self, client, event):
        if is_yateam_uid(event.uid):
            return
        try:
            if not self.dry_run:
                if self.backoff.is_active() and client.service_name in self.failed_services:
                    backoff = self.backoff.get()
                    log.debug('Sleeping for %f seconds before next try (%s)', backoff, client.service_name)
                    time.sleep(backoff)
                if client.service_name == 'disk_2':
                    client.notify(event)
                else:
                    if client.service_name == 'plus_notifiers':
                        current_time = time.time()
                        if current_time - int(event.timestamp) < 20:
                            time.sleep(20)
                    event.name = self.get_event_name_for_event(event)
                    client.notify(event)
                if client.service_name in self.failed_services:
                    self.failed_services.remove(client.service_name)
                    self.backoff.reset()
            log.debug('Successfully notified %s. Uid=%s. Source event=%s',
                      client.service_name, event.uid, event.__class__.__name__)
        except BaseNotifyError as e:
            self.failed_services.add(client.service_name)
            self.backoff.update()
            raise ServiceHandlerException(e)
        except BlackboxTemporaryError as e:
            raise ServiceHandlerException(e)
