# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.blackbox.exceptions import BaseBlackboxError
from passport.backend.core.builders.datasync_api.exceptions import (
    DatasyncApiAuthorizationInvalidError,
    DatasyncApiObjectNotFoundError,
    DatasyncApiPermanentError,
    DatasyncApiTemporaryError,
)
from passport.backend.core.builders.edadeal import (
    EdadealPermanentError,
    EdadealTemporaryError,
)
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.core.eav_type_mapping import get_attr_type

from .exceptions import (
    BaseNotifyError,
    NotifyRequestError,
    NotifyTemporaryError,
)


log = logging.getLogger(__name__)


class NotifyClient(BaseBuilder, JsonParserMixin):
    base_error_class = BaseNotifyError
    temporary_error_class = NotifyTemporaryError
    parser_error_class = NotifyRequestError
    accept_empty_response = True

    def __init__(self, service_name, url, timeout=5.0, retries=5, useragent=None,
                 blackbox=None, disk=None, edadeal=None):
        super(NotifyClient, self).__init__(
            url=url,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
        )
        self.service_name = service_name
        self.blackbox = blackbox
        self.disk = disk
        self.edadeal = edadeal

    def check_response_for_errors(self, data, raw_response):
        if raw_response.status_code != 200 or data.get('status') != 'ok':
            raise BaseNotifyError('Service %s invalid response. status=%s, content=%s' %
                                  (self.service_name, raw_response.status_code, raw_response.content))

    def notify(self, event):
        data = {
            'v': 1,
            'uid': event.uid,
            'event': event.name,
            'timestamp': event.timestamp,
        }
        try:
            return self._request_with_retries_simple(
                url_suffix=self.url,
                method='POST',
                error_detector=self.check_response_for_errors,
                parser=self.parse_json,
                data=json.dumps(data),
                headers={'Content-Type': 'application/json'},
            )
        except BaseNotifyError as e:
            log.error("Couldn't notify %s. Request: url=%s data=%s Error: %s", self.service_name, self.url, data, e)
            raise


class DiskNotifyClient(NotifyClient):
    def notify(self, event):
        data = {
            'v': 1,
            'uid': event.uid,
            'event': event.name,
            'timestamp': event.timestamp,
        }
        try:
            return self.disk._request_with_retries_simple(
                url_suffix=self.url,
                method='POST',
                error_detector=self.check_response_for_errors,
                parser=self.parse_json,
                data=json.dumps(data),
                headers={'Content-Type': 'application/json'},
            )
        except BaseNotifyError as e:
            log.error("Couldn't notify %s. Request: url=%s data=%s Error: %s", self.service_name, self.url, data, e)
            raise
        except (DatasyncApiTemporaryError, DatasyncApiAuthorizationInvalidError) as e:
            log.error("Couldn't notify %s. Request: url=%s data=%s Error: %s", self.service_name, self.url, data, e)
            raise NotifyTemporaryError()
        except (DatasyncApiObjectNotFoundError, DatasyncApiPermanentError) as e:
            log.error("Couldn't notify %s (won't retry). Request: url=%s data=%s Error: %s", self.service_name, self.url, data, e)
            return


class PlusNotifyClient(NotifyClient):
    def notify(self, event):
        try:
            # ToDo: стоит поменять на 1015 аттрибут когда звезды встанут на свои места
            account_plus_enabled_type = str(get_attr_type('account.plus.enabled'))
            result = self.blackbox.userinfo(
                uid=event.uid,
                attributes=[account_plus_enabled_type],
                need_display_name=False,
                need_public_name=False,
                need_aliases=False,
            ).get('attributes', {}).get(account_plus_enabled_type, '0')
            if result == '0':
                self.disk.plus_unsubscribe(event.uid, 'yandex_plus', 'yandex_plus_10gb')
                self.edadeal.update_plus_status(event.uid, is_active=False)
            elif result == '1':
                self.disk.plus_subscribe(event.uid, 'yandex_plus', 'yandex_plus_10gb')
                self.edadeal.update_plus_status(event.uid, is_active=True)

        except BaseBlackboxError as e:
            log.error("Couldn't get to blackbox. Error: %s", e)
            raise
        except BaseNotifyError as e:
            log.error("Couldn't notify %s. Request: url=%s Error: %s", self.service_name, self.url, e)
            raise
        except (DatasyncApiTemporaryError, DatasyncApiAuthorizationInvalidError, EdadealTemporaryError) as e:
            log.error("Couldn't notify %s. Request: url=%s Error: %s", self.service_name, self.url, e)
            raise NotifyTemporaryError()
        except (DatasyncApiObjectNotFoundError, DatasyncApiPermanentError, EdadealPermanentError) as e:
            log.error("Couldn't notify %s (won't retry). Request: url=%s Error: %s", self.service_name, self.url, e)
            return
