# -*- coding: utf-8 -*-
import logging

from passport.backend.core.avatars import upload_avatar
from passport.backend.core.builders.avatars_mds_api import (
    AvatarsMdsApiBadImageFormatError,
    AvatarsMdsApiInvalidFileSizeError,
    AvatarsMdsApiInvalidImageSizeError,
    AvatarsMdsApiPermanentError,
    AvatarsMdsApiTemporaryError,
)
from passport.backend.core.builders.blackbox import Blackbox
from passport.backend.core.builders.passport import (
    Passport,
    PassportAccountNotFoundError,
    PassportPermanentError,
    PassportTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.models.account import Account
from passport.backend.core.utils.decorators import cached_property
from passport.backend.logbroker_client.avatars.events import UploadAvatarEvent
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk


log = logging.getLogger(__name__)


class AvatarsHandlerException(HandlerException):
    pass


class AvatarsHandler(BaseHandler):
    handler_name = 'avatars'

    def __init__(self, config, **kwargs):
        super(AvatarsHandler, self).__init__(config=config, **kwargs)
        events_classes = [UploadAvatarEvent]
        self.filter = BasicFilter(events_classes)

    def parse_message(self, message):
        return self.filter.filter(message)

    def process(self, header, data):
        message = MessageChunk(header, data)
        events = self.get_message_entries(message)
        for event in events:
            self.process_event(event)
        return True

    @cached_property
    def blackbox(self):
        return Blackbox(
            settings.BLACKBOX_URL,
            use_tvm=settings.BLACKBOX_USE_TVM,
        )

    @cached_property
    def passport(self):
        return Passport(
            use_tvm=settings.PASSPORT_USE_TVM,
        )

    def process_event(self, event):
        try:
            if event.name == 'upload_by_url':
                if event.skip_if_set:
                    userinfo = self.blackbox.userinfo(
                        uid=event.uid,
                        attributes=[],
                        dbfields=[],
                        need_display_name=True,
                        need_public_name=False,
                        need_aliases=False,
                    )
                    account = Account().parse(userinfo)
                    if not account.person.is_avatar_empty:
                        log.info('Account (uid=%s) already has avatar, skipping', event.uid)
                        return

                avatar_id = upload_avatar(event.uid, avatar_url=event.avatar_to_upload)
                self.passport.set_default_avatar(event.uid, avatar_id, event.user_ip)
                log.debug(
                    'Successfully uploaded avatar. Uid=%s, url=%s, initiated from IP %s, skip_if_set=%s',
                    event.uid,
                    event.avatar_to_upload,
                    event.user_ip,
                    event.skip_if_set,
                )
        except (UnknownUid, PassportAccountNotFoundError):
            log.info('Account %s not found', event.uid)
            return
        except (
            AvatarsMdsApiInvalidImageSizeError,
            AvatarsMdsApiInvalidFileSizeError,
            AvatarsMdsApiBadImageFormatError,
        ) as e:
            log.info('Avatar not suitable for uid=%s (won\'t retry): %s', event.uid, e)
            return
        except (PassportTemporaryError, AvatarsMdsApiTemporaryError) as e:
            log.error('Error while processing avatar for uid=%s: %s', event.uid, e)
            raise AvatarsHandlerException(e)
        except (PassportPermanentError, AvatarsMdsApiPermanentError) as e:
            log.error('Error while processing avatar for uid=%s (won\'t retry): %s', event.uid, e)
            return
