from functools import cached_property
import logging
import time
from typing import (
    Iterable,
    List,
    Tuple,
)

from passport.backend.core.am_pushes.common import (
    get_am_capabilities_manager,
    Platforms,
)
from passport.backend.core.am_pushes.push_request import (
    APP_TARGETING_TYPE_TO_NAME,
    AppTargetingTypes,
    PushRequest,
    PushRequestRecipient,
    SUBSCRIPTION_SOURCE_TO_NAME,
    SubscriptionSources,
)
from passport.backend.core.am_pushes.subscription_manager import (
    CapsFilter,
    DeviceIdFilter,
    DeviceSubscriptions,
    get_pushes_subscription_manager,
    PlatformFilter,
    SubscriptionRatingPerDeviceProcessor,
)
from passport.backend.core.builders.push_api import BasePushApiError
from passport.backend.core.builders.push_api.push_api import (
    get_push_api,
    Subscription,
)
from passport.backend.core.logging_utils.loggers.push import PushLogger
from passport.backend.logbroker_client.core.handlers.protobuf import BaseProtobufHandler
from passport.backend.utils.common import noneless_dict


log = logging.getLogger('logbroker')
# Сколько подписок мы максимально перебираем на тему живости на каждый девайс
MAX_SUBSCRIPTION_LOOKUP_DEPTH = 10


class ChallengePushesHandler(BaseProtobufHandler):
    handler_name = 'challenge_pushes'

    def __init__(
        self, subscription_rating, with_test_subscriptions=False,
        max_retries_for_transport=3, *args, **kwargs
    ):
        self.app_priority = subscription_rating.get('app', [])
        self.with_test_subscriptions = with_test_subscriptions
        self.max_retries_for_transport = max_retries_for_transport

        super(ChallengePushesHandler, self).__init__(*args, **kwargs)

    @property
    def push_api(self):
        return get_push_api()

    @property
    def am_caps_manager(self):
        return get_am_capabilities_manager()

    @cached_property
    def push_log(self):
        return PushLogger()

    def _make_am_capabilities_filter(
        self, recipient: PushRequestRecipient,
    ) -> CapsFilter:
        return CapsFilter(recipient.required_am_capabilities)

    def _make_platform_filter(
        self, recipient: PushRequestRecipient,
    ) -> PlatformFilter:
        return PlatformFilter(recipient.required_platforms)

    def _recipient_to_subscriptions(
        self, recipient: PushRequestRecipient, push_service: str, event: str,
    ) -> Tuple[Iterable[Subscription], bool]:
        if recipient.app_targeting_type in (
            AppTargetingTypes.GLOBAL_WITH_CAPS,
            AppTargetingTypes.CLIENT_DECIDES,
            AppTargetingTypes.ONE_APP_PER_DEVICE,
        ):
            filters = []

            if recipient.required_platforms:
                filters.append(PlatformFilter(recipient.required_platforms))

            if recipient.required_am_capabilities:
                filters.append(CapsFilter(recipient.required_am_capabilities))

            if recipient.device_ids:
                filters.append(DeviceIdFilter(recipient.device_ids))

            subscription_manager = get_pushes_subscription_manager(
                uid=recipient.uid,
                push_service=push_service,
                event=event,
            )

            if recipient.subscription_source == SubscriptionSources.YAKEY:
                subscriptions = subscription_manager.yakey_compatible_subscriptions
            elif recipient.require_trusted_device:
                subscriptions = subscription_manager.trusted_subscriptions
            else:
                subscriptions = subscription_manager.am_compatible_subscriptions

            subscriptions = subscription_manager.filter_subscriptions(
                subscriptions=subscriptions,
                filters=filters,
            )
            if self.with_test_subscriptions and subscription_manager.test_subscriptions:
                subscriptions = list(subscriptions) + subscription_manager.test_subscriptions
        else:
            subscriptions = None

        is_silent = recipient.app_targeting_type == AppTargetingTypes.CLIENT_DECIDES
        return subscriptions, is_silent

    def _set_subscriptions_filter(
        self, body: dict, subscription_ids: List[int],
    ):
        body['subscriptions'] = [{'subscription_id': subscription_ids}]

    @staticmethod
    def _version_as_str(version):
        return '.'.join(map(str, version)) if version is not None else None

    def _make_push_body(
        self, request: PushRequest, recipient: PushRequestRecipient,
        silent: bool,
    ) -> dict:
        """
        Генерирует тело пуша в виде словаря
            payload: произвольные поля данных для АМ
            repack: специальные по платформе, плюс правила перепаковки
                apns: (apple)
                    aps:
                        alert: поля тела сообщения для отрисовки в ОС
                            (только для не-silent пушей)
                        "content-available": 1 - передаётся всегда
                repack_payload: (правила упаковки payload)
                    упаковываем все ключи payload,
                    плюс мапим "event" как "e"

                fcm: (google)
                    notification: поля тела сообщения для отрисовки в ОС
                        (только для не-silent пушей)
                    repack_payload: (правила упаковки payload)
                        просто упаковываем все ключи payload
        """
        payload = noneless_dict(
            passp_am_proto=request.AM_PROTO_VERSION,
            push_service=request.push_service,
            event_name=request.event_name,
            push_id=request.push_id,
            is_silent=silent,
            uid=recipient.uid,
            title=request.title,
            body=request.body,
            subtitle=request.subtitle,
            webview_url=request.webview_url,
            require_web_auth=request.require_web_auth,
            **(request.extra_data or {})
        )

        # Apple
        repack_apns = dict(
            repack_payload=[dict(e='event_name')] + list(payload.keys()),
            aps={'content-available': 1},
        )
        if not silent:
            # Для Android не заполняем: андроидные пуши всегда ходят как silent и при необходимости рендерятся уже АМом
            repack_apns['aps'].update(
                alert=noneless_dict(
                    title=request.title,
                    body=request.body,
                    subtitle=request.subtitle,
                ),
            )

        # Google
        repack_fcm = dict(
            repack_payload=list(payload.keys()),
        )

        min_am_versions = {
            'min_am_version_' + platform.name: self._version_as_str(
                self.am_caps_manager.get_min_am_version_by_caps(
                    platform,
                    recipient.required_am_capabilities,
                ),
            )
            for platform in Platforms
        }
        payload.update(noneless_dict(**min_am_versions))
        if payload.get('min_am_version_android'):
            repack_fcm['repack_payload'].append(dict(min_am_version='min_am_version_android'))
        if payload.get('min_am_version_ios'):
            repack_apns['repack_payload'].append(dict(min_am_version='min_am_version_ios'))

        return dict(
            payload=payload,
            repack=dict(
                apns=repack_apns,
                fcm=repack_fcm,
            ),
        )

    def _generate_sub_list(self, device_subscriptions: DeviceSubscriptions):
        result = []
        for subscription in device_subscriptions.values():
            if not subscription.is_finished:
                try:
                    result.append(subscription.current)
                except StopIteration:
                    pass

        return result

    def _is_send_result_ok(self, result: dict) -> bool:
        return result.get('code') == 200

    def _is_send_result_temporary_error(self, result: dict) -> bool:
        return 500 <= result.get('code', 0) <= 599

    def _send_one_app_per_device(
        self, request: PushRequest, recipient: PushRequestRecipient,
        push_body: dict, subscriptions_list: Iterable[Subscription],
    ):
        app_priority = recipient.custom_app_priority or self.app_priority
        generator = SubscriptionRatingPerDeviceProcessor(app_priority)
        device_subscriptions = generator.generate(subscriptions_list)

        log.debug('Begin sending loop')
        for _ in range(MAX_SUBSCRIPTION_LOOKUP_DEPTH):
            if not device_subscriptions:
                log.debug('Sending loop finished')
                break
            cur_subscriptions = self._generate_sub_list(device_subscriptions)
            subscription_ids = [s.id for s in cur_subscriptions]
            self._set_subscriptions_filter(push_body, subscription_ids)
            log.debug('Sending to subscriptions {}'.format(subscription_ids))
            try:
                res = self.push_api.send(
                    event=request.event_name,
                    payload=push_body,
                    user=recipient.uid,
                    sync=True,
                )
                if not res:
                    log.warning('Unexpected empty result of sync send call')
                    break
                for device_id, subscriptions in list(device_subscriptions.items()):
                    cur_sub = subscriptions.current
                    cur_res = res.get(cur_sub.id)
                    if cur_res is None:
                        log.warning(
                            'Subscription {} device {} missing in sync send call result'.format(
                                cur_sub.id, device_id,
                            ),
                        )
                        device_subscriptions[device_id].set_finished()
                        status = 'error'
                        details = 'Result missing'
                    elif self._is_send_result_ok(cur_res):
                        log.debug(
                            'Successfully sent to subscription {} device {}'.format(
                                cur_sub.id, device_id,
                            )
                        )
                        device_subscriptions[device_id].set_finished()
                        status = 'ok'
                        details = str(cur_res)
                    else:
                        is_temporary = self._is_send_result_temporary_error(cur_res)
                        log.debug('{} error sending to subscription {} device {}: {}'.format(
                            'Temporary' if is_temporary else 'Critical',
                            cur_sub.id, device_id, cur_res,
                        ))
                        if (
                            is_temporary and
                            device_subscriptions[device_id].num_tries + 1 < self.max_retries_for_transport
                        ):
                            device_subscriptions[device_id].num_tries += 1
                        else:
                            device_subscriptions[device_id].set_next()
                            if device_subscriptions[device_id].is_finished:
                                log.debug('No more subscriptions for device {}'.format(device_id))
                            else:
                                log.debug('Next try for device {}'.format(device_id))
                        status = 'error'
                        details = str(cur_res)
                    self.push_log.log(
                        subscription_id=cur_sub.id,
                        device_id=cur_sub.device,
                        app_id=cur_sub.app,
                        status=status,
                        details=details,
                    )
            except BasePushApiError as ex:
                error = str(ex)
                log.exception(
                    'Error sending push for uid {}: `{}`'.format(
                        recipient.uid,
                        error,
                    ),
                )

    def _process_am_recipient(
        self, request: PushRequest, recipient: PushRequestRecipient,
    ):
        subscriptions, is_silent = self._recipient_to_subscriptions(
            recipient=recipient,
            push_service=request.push_service,
            event=request.event_name,
        )
        push_body = self._make_push_body(request, recipient, is_silent)
        if subscriptions is not None:
            if subscriptions:
                log.debug('Suitable subscriptions for uid {}: {}'.format(
                    recipient.uid, [s.id for s in subscriptions],
                ))
            else:
                log.debug(
                    'No suitable subscriptions found for '
                    'uid={} selection method={} subscription_source={}'.format(
                        recipient.uid,
                        APP_TARGETING_TYPE_TO_NAME.get(
                            recipient.app_targeting_type,
                            recipient.app_targeting_type,
                        ),
                        SUBSCRIPTION_SOURCE_TO_NAME.get(
                            recipient.subscription_source,
                            recipient.subscription_source,
                        ),
                    ),
                )
                # Логируем, что пуш не отправлен
                self.push_log.log(
                    subscription_id='n/a',
                    status='not_sent',
                    details='No subscriptions filtered',
                )
                return
        else:
            log.debug('No subscription filter is needed for {}'.format(recipient.uid))

        log.info('Sending push for uid {}'.format(recipient.uid))
        if recipient.app_targeting_type == AppTargetingTypes.ONE_APP_PER_DEVICE:
            return self._send_one_app_per_device(request, recipient, push_body, subscriptions)
        else:
            if subscriptions is not None:
                subscription_ids = [s.id for s in subscriptions]
                self._set_subscriptions_filter(push_body, subscription_ids)
        status = 'ok'
        details = None
        try:
            self.push_api.send(
                event=request.event_name,
                payload=push_body,
                user=recipient.uid,
            )
        except BasePushApiError as ex:
            details = str(ex)
            log.exception(
                'Error sending push for uid {}: `{}`'.format(
                    recipient.uid,
                    details,
                ),
            )

        if subscriptions is None:
            self.push_log.log(
                subscription_id='GLOBAL',
                status=status,
                details=details,
            )
        else:
            for subscription in subscriptions:
                self.push_log.log(
                    subscription_id=subscription.id,
                    device_id=subscription.device,
                    app_id=subscription.app,
                    status=status,
                    details=details,
                )

    def process_am_push_message(self, message):
        request = PushRequest.from_proto(message.push_message_request)
        log.debug('Got AM push task {} for uid {}'.format(
            request.push_id, ', '.join(str(r.uid) for r in request.recipients),
        ))
        if request.expire_time and request.expire_time < time.time():
            log.warning('AM push task {} expired {} sec ago'.format(
                request.push_id, time.time() - request.expire_time,
            ))
            return
        for recipient in request.recipients:
            with self.push_log.make_context(
                push_id=request.push_id,
                uid=recipient.uid,
                push_service=request.push_service,
                push_event=request.event_name,
                context=recipient.context,
            ):
                self._process_am_recipient(request, recipient)

    def process_message(self, header, message):
        if not message.HasField('push_message_request'):
            log.warning('Got unsupported legacy message')
        else:
            self.process_am_push_message(message)
