# -*- coding: utf-8 -*-
import logging
import re
import socket
import time

from passport.backend.logbroker_client.core.logbroker.client import LogbrokerMeta
from passport.backend.logbroker_client.core.runner.emitters.base import Emitter
from six import string_types


log = logging.getLogger(__name__)


class PartitionsCountEmitter(Emitter):
    def __init__(self, balancer_host, client, ident, workers_count, per_host_config=None, data_port=None):
        super(PartitionsCountEmitter, self).__init__()
        if isinstance(ident, string_types):
            idents = [ident]
        else:
            idents = ident
        self.idents = idents
        self.client = client
        self.INTERRUPTED = False
        self.host_config = self.get_host_config(per_host_config)
        self.workers_count = self.host_config.get('workers_count', workers_count)
        self.dc = self.host_config.get('dc')
        if not self.dc:
            raise ValueError('No dc parameter is stated')
        self.balancer_host = balancer_host % dict(dc=self.dc)
        self.data_port = data_port

    def handle_sigint(self, signum, frame):
        self.INTERRUPTED = True

    def split_partitions(self, partitions, workers_count):
        parts = {x: [] for x in range(workers_count)}
        for i, partition in enumerate(partitions):
            parts[i % workers_count].append(partition)
        return parts

    def get_host_config(self, per_host_config):
        for server_pattern, host_config in per_host_config.items():
            if re.match(server_pattern, socket.gethostname()):
                return host_config

    def filter_topics(self, partitions):
        topics = set(x.split(':')[0] for x in partitions)
        if self.host_config.get('topics'):
            filtered_topics = []
            for topic_pattern in self.host_config.get('topics'):
                for topic in topics:
                    if re.match(topic_pattern, topic, re.IGNORECASE):
                        filtered_topics.append(topic)
            return filtered_topics
        else:
            return topics

    def filter_partitions(self, partitions, topics):
        result_partitions = []
        for partition in partitions:
            topic = partition.split(':')[0]
            if topic in topics:
                result_partitions.append(partition)
        return result_partitions

    def loop(self):
        client = LogbrokerMeta(self.balancer_host, self.client)
        log.info(
            'Load with config: balancer_host=%s client=%s dc=%s',
            self.balancer_host,
            self.client,
            self.dc,
        )
        hosts = client.hosts_info(self.dc)
        log.info('Hosts in dc=%s: %s', self.dc, hosts)
        log.info('Load topics for idents: %s', self.idents)
        partitions = []
        for ident in self.idents:
            ident_partitions = client.show_parts(ident=ident)
            partitions.extend(ident_partitions)
        filtered_topics = self.filter_topics(partitions)
        if not filtered_topics:
            log.warning('No topics to read from. Set correct `topics_filters` setting in config file!')
            self.INTERRUPTED = True
            return

        log.info('Topics: %s' % filtered_topics)
        filtered_partitions = self.filter_partitions(partitions, filtered_topics)
        log.info('Partitions for idents=%s: %s', self.idents, filtered_partitions)

        partitions_count = self.host_config.get('read_partitions_count') or len(filtered_partitions)
        filtered_partitions = filtered_partitions[:partitions_count]
        splits = self.split_partitions(filtered_partitions, self.workers_count)
        log.info('Split %s partitions for %s workers: %s', len(filtered_partitions), self.workers_count, splits)
        for partitions_chunk in splits.values():
            if not partitions_chunk:
                continue
            log.info('Created task with partitions: %s partitions', filtered_partitions)
            self.tasks_queue.put(
                {
                    'hosts': hosts,
                    'client': self.client,
                    'partitions_count': len(partitions_chunk),
                    'topics': filtered_topics,
                    'data_port': self.data_port,
                },
            )
        while not self.INTERRUPTED:
            time.sleep(1)
        log.info('Emitter terminated')
