# -*- coding: utf-8 -*-
import logging
import re
import socket
import time

from passport.backend.logbroker_client.core.runner.emitters.base import Emitter


log = logging.getLogger('logbroker')


class PartitionsCountNativeEmitter(Emitter):
    """ Генератор заданий для родного logbroker клиента (logbroker SDK) """
    def __init__(
        self,
        per_host_config,
        data_port,
        client_id,
        credentials_config,
        workers_count,
        connect_timeout=None,
        ca_cert=None,
        ssl=True,
        decompress=False,
        use_client_locks=False,
    ):
        """
        :param per_host_config: dict конфигураций хостов
            ключ - regex для матчинга hostname
            значение - dict конфигурации хоста:
                targets: list целей. Каждая цель - dict:
                    host - хост logbroker-эндпойнта
                    topic - имя топика
                    ssl - optional, переопределить включение ssl для цели
                    data_port - optional, переопределить порт для цели
        :param data_port: порт данных
        :param client_id: имя client_id (logbroker consumer)
        :param credentials_config: dict с обязательным полем 'credentials_type'
            Остальные поля зависят от credentials_type:
                - 'credentials_type': 'oauth_token'
                    не требуется дополнительных полей, oauth token ожидается в
                    переменной окружения LB_TOKEN
                - 'credentials_type': 'tvm'
        :param workers_count: кол-во воркеров
        :param connect_timeout: float, таймаут соединения с lb
        :param ca_cert: optional, альтернативный ca-сертификат
        :param ssl: optional, default=True, включить ssl
        :param decompress: optional, default=False, распаковывать сжатие
        :param use_client_locks: optional, default=False, включает режим
            явного присвоения партиций. Полезно для отладки чтения
        """
        super(PartitionsCountNativeEmitter, self).__init__()
        self.host_config = self.get_host_config(per_host_config)
        self.workers_count = self.host_config.get('workers_count', workers_count)
        self.data_port = data_port
        self.client_id = client_id
        self.credentials_config = credentials_config
        self.connect_timeout = connect_timeout
        self.ca_cert = ca_cert
        self.ssl = ssl
        self.decompress = decompress
        self.use_client_locks = use_client_locks
        if self.workers_count < len(self.host_config['targets']):
            # Воркер синхронный, умеет держать только одну сессию
            raise ValueError('workers count cannot be less than targets count')
        self.INTERRUPTED = False

    def get_host_config(self, per_host_config):
        for server_pattern, host_config in per_host_config.items():
            if re.match(server_pattern, socket.gethostname()):
                return host_config

    def handle_sigint(self, signum, frame):
        self.INTERRUPTED = True

    def emit(self):
        targets = self.host_config['targets']
        for i in range(self.workers_count):
            target = targets[i % len(targets)]
            task = {
                'host': target['host'],
                'port': target.get('data_port', self.data_port),
                'ssl': target.get('ssl', self.ssl),
                'ca_cert': self.ca_cert,
                'client_id': target.get('client_id', self.client_id),
                'credentials_config': target.get('credentials_config', self.credentials_config),
                'topic': target['topic'],
                'decompress': self.decompress,
                'use_client_locks': self.use_client_locks,
            }
            log.info('Emitting task {}'.format(task))
            if self.connect_timeout is not None:
                task['connect_timeout'] = self.connect_timeout
            self.tasks_queue.put(task)

    def loop(self):
        log.info(
            'Starting emitter with config: {}'.format(
                {
                    'host_config': self.host_config,
                    'workers_count': self.workers_count,
                    'data_port': self.data_port,
                    'client_id': self.client_id,
                    'connect_timeout': self.connect_timeout,
                    'credentials_config': self.credentials_config,
                    'decompress': self.decompress,
                    'use_client_locks': self.use_client_locks,
                    'ssl': self.ssl,
                    'ca_cert': self.ca_cert,
                },
            ),
        )
        self.emit()

        while not self.INTERRUPTED:
            time.sleep(1)
        log.info('Emitter terminated')
