# -*- coding: utf-8 -*-
import logging
import signal
from threading import Thread
import time

from passport.backend.logbroker_client.core.consumers.exceptions import ExitException
from passport.backend.logbroker_client.core.consumers.simple.worker import LogbrokerWorker
from passport.backend.logbroker_client.core.handlers.exceptions import (
    HandlerException,
    ProtocolError,
)
from passport.backend.logbroker_client.core.logbroker.client import BaseLogbrokerClientException
from passport.backend.logbroker_client.core.logbroker.native_client import (
    ConnectError,
    NativeLogbrokerConsumer,
)
from passport.backend.logbroker_client.core.utils import LogPrefixManager


log = logging.getLogger('logbroker')


class HeartbeatThread(Thread):
    def __init__(self, req_queue, resp_queue, client):
        super(HeartbeatThread, self).__init__()
        self.req_queue = req_queue
        self.resp_queue = resp_queue
        self.client = client

    def run(self):
        while True:
            try:
                _ = self.req_queue.get()
                log.debug('Heartbeat info requested')
                self.resp_queue.put(self.client.details)
            except Exception:
                log.exception('Unhandled exception in worker heartbeat thread')
                time.sleep(5)


class NativeLogbrokerWorker(LogbrokerWorker):
    current_task = None
    current_config = None

    """ Воркер на базе нативного logbroker-клиента (python logbroker SDK)"""
    def __init__(self, bind_signals=False, *args, **kwargs):
        super(NativeLogbrokerWorker, self).__init__(*args, **kwargs)

        if bind_signals:
            signal.signal(signal.SIGINT, self.handle_sigint)
            signal.signal(signal.SIGTERM, signal.SIG_IGN)

    def handle_sigint(self, signum, frame):
        log.warning('Got SIGINT, terminating actor')
        self.INTERRUPTED = True

    def create_client(self, task):
        return NativeLogbrokerConsumer(
            host=task['host'],
            port=task['port'],
            ca_cert=task.get('ca_cert'),
            client_id=task['client_id'],
            credentials_config=task['credentials_config'],
            topic=task['topic'],
            decompress=task['decompress'],
            use_client_locks=task.get('use_client_locks', False),
            partition_group=task.get('partition_group'),
            connect_timeout=task.get('connect_timeout', 0.3),
            max_count=task.get('max_count'),
            is_interrupted_callback=self._is_interrupted,
        )

    def connect_loop(self, client):
        while not self.INTERRUPTED:
            try:
                client.start()
                log.info('Started logbroker client')
                break
            except BaseLogbrokerClientException as e:
                log.error('(connect) %s: %s', e.__class__.__name__, e)
                time.sleep(1.0)

    def reconnect_loop(self, client):
        while not self.INTERRUPTED:
            log.info('Trying to reconnect to logbroker')
            try:
                client.start_consumer()
                log.info('Reconnected to logbroker')
                break
            except BaseLogbrokerClientException as e:
                log.error('(reconnect) %s: %s', e.__class__.__name__, e)
                time.sleep(1.0)

    def _is_interrupted(self):
        return self.INTERRUPTED

    def _handle_message(self, topic, partition, message, decompressed_data):
        extra_fields = {
            item.key: item.value for item in message.meta.extra_fields.items
        }
        header = {
            'topic': topic,
            'partition': partition,
            'offset': message.offset,
            'path': extra_fields.get('file', extra_fields.get('path')),
            'server': extra_fields.get('server'),
            'ident': extra_fields.get('ident'),
            'logtype': extra_fields.get('logtype'),
            'message_class': extra_fields.get('message_class'),
            'passport_protocol_v': extra_fields.get('passport_protocol_v'),
            'extra_fields': extra_fields,
        }
        while True:
            try:
                with LogPrefixManager.user_context():
                    self.handler.process(header, decompressed_data)
                break
            except ProtocolError as err:
                log.exception('Handler protocol error: {}'.format(err))
                break
            except HandlerException as e:
                log.warning('%s: %s', e.__class__.__name__, e)
                time.sleep(0.1)
            except Exception:
                log.exception('Unhandled exception in handler')
                time.sleep(0.1)

    def read_loop(self, client):
        while not self.INTERRUPTED:
            try:
                client.handle_message(self._handle_message)
            except ConnectError as e:
                log.error('Logbroker connection error: {}'.format(e))
                self.reconnect_loop(client)
            except BaseLogbrokerClientException as e:
                log.warning('%s: %s', e.__class__.__name__, e)
                time.sleep(0.1)
            except ExitException:
                log.info('Worker terminated')
                break

    def run_task(self, task, heartbeat_req_queue=None, heartbeat_resp_queue=None):
        log.info('Running task %s', task)
        client = self.create_client(task)

        if heartbeat_req_queue and heartbeat_resp_queue:
            heartbeat_thread = HeartbeatThread(
                req_queue=heartbeat_req_queue,
                resp_queue=heartbeat_resp_queue,
                client=client,
            )
            heartbeat_thread.daemon = True
            heartbeat_thread.start()

        self.connect_loop(client)

        with client:
            self.read_loop(client)

    def loop(self):
        log.info('Run worker')
        try:
            task = self.read_task()
        except IOError:
            log.warning("Couldn't receive task from queue")
            return
        self.run_task(task)
