# -*- coding: utf-8 -*-
import logging
import time
import weakref

from passport.backend.logbroker_client.core.consumers.exceptions import ExitException
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.logbroker.client import (
    BaseLogbrokerClientException,
    LogbrokerConsumer,
)
from passport.backend.logbroker_client.core.runner.workers.base import Worker
from passport.backend.logbroker_client.core.utils import importobj


log = logging.getLogger(__name__)
common_log = logging.getLogger('delta')


class SimpleHandlerDecorator(BaseHandler):
    def __init__(self, worker, handler, config):
        super(SimpleHandlerDecorator, self).__init__(config)
        self.worker = worker
        self.handler = handler

    def process(self, header, data):
        return self.handler.process(header, data)

    def flush(self, force=False):
        return self.handler.flush(force)

    def parse_message(self, message, **kwargs):
        raise NotImplementedError()


class LogbrokerWorker(Worker):
    INTERRUPTED = False

    def __init__(self, handler, config):
        super(LogbrokerWorker, self).__init__()
        handler_cls = importobj(handler.get('class'))
        handler_args = handler.get('args')
        handler_args.update({'config': config})
        try:
            handler = handler_cls(**handler_args)
        except TypeError as err:
            import inspect
            raise TypeError('{} handler args: {}, handler arg specs: {}'.format(
                err,
                list(handler_args.keys()),
                inspect.formatargspec(*inspect.getargspec(handler_cls.__init__)),
            ))
        self.handler = SimpleHandlerDecorator(
            worker=weakref.proxy(self),
            handler=handler,
            config=config,
        )

    def handle_sigint(self, signum, frame):
        self.INTERRUPTED = True

    def read_task(self):
        return self.tasks_queue.get()

    def create_client(self, task):
        return LogbrokerConsumer(
            task['hosts'],
            task['client'],
            task['topics'],
            task['partitions_count'],
            task['data_port'],
        )

    def loop(self):
        log.info('Run worker')
        try:
            task = self.read_task()
        except IOError:
            log.warning("Couldn't receive task from queue")
            return

        log.info('Got task %s', task)
        while not self.INTERRUPTED:
            client = self.create_client(task)
            try:
                client.read_unpacked(self.handler)
            except BaseLogbrokerClientException as e:
                log.warning('%s: %s', e.__class__.__name__, e)
                time.sleep(0.1)
            except HandlerException as e:
                log.warning('%s: %s', e.__class__.__name__, e)
                time.sleep(0.1)
            except ExitException:
                log.info('Worker terminated')
                break


class AutoBalancedLogbrokerWorker(LogbrokerWorker):
    def create_client(self, task):
        return LogbrokerConsumer(
            task['hosts'],
            task['client'],
            task['topics'],
            task['partitions_count'],
            task['data_port'],
            suggest_timeout=120,
            suggest_balance='all',
        )
