# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.library.historydbloader.historydb.parser import ParserType


log = logging.getLogger(__name__)


class Event(object):
    @classmethod
    def is_this_event(cls, parser_type, event):
        raise NotImplementedError()

    def __init__(self, name, timestamp=None):
        self.name = name
        self.timestamp = int(float(timestamp)) if timestamp else timestamp

    def to_dict(self):
        return {
            'name': self.name,
            'timestamp': self.timestamp,
        }

    def to_json(self):
        return json.dumps(self.to_dict())


class UserEvent(Event):
    def __init__(self, name, uid, timestamp=None):
        super(UserEvent, self).__init__(name, timestamp)
        self.uid = uid

    def to_dict(self):
        result = super(UserEvent, self).to_dict()
        result['uid'] = self.uid
        return result

    @classmethod
    def is_this_event(cls, parser_type, event):
        raise NotImplementedError()


class AccountDisableEvent(UserEvent):
    # событие блоикировки пользователя
    NAME = 'account.disable'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('name') == 'info.ena' and
                event.get('value') == '0'):
            return cls(event.get('uid'), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AccountDisableEvent, self).__init__(self.NAME, uid, timestamp)


class AccountEnableEvent(UserEvent):
    NAME = 'account.enable'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('name') == 'info.ena' and
                event.get('value') == '1'):
            return cls(event.get('uid'), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AccountEnableEvent, self).__init__(self.NAME, uid, timestamp)


class AccountRemovalDisableEvent(UserEvent):
    NAME = 'account.removal_disable'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('name') == 'info.ena' and
                event.get('value') == '2'):
            return cls(event.get('uid'), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AccountRemovalDisableEvent, self).__init__(self.NAME, uid, timestamp)


class AccountGlogoutEvent(UserEvent):
    # событие глобального разлогина
    NAME = 'account.glogout'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'info.glogout':
            return cls(event.get('uid'), event.get('value') or event.get('timestamp'))

    def __init__(self, uid, timestamp):
        super(AccountGlogoutEvent, self).__init__(self.NAME, uid, timestamp)


class TokensRevokedEvent(UserEvent):
    NAME = 'account.tokens_revoked'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'info.tokens_revoked':
            return cls(event.get('uid'), event.get('value') or event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(TokensRevokedEvent, self).__init__(self.NAME, uid, timestamp)


class SessionsRevokedEvent(UserEvent):
    NAME = 'account.web_sessions_revoked'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'info.web_sessions_revoked':
            return cls(event.get('uid'), event.get('value') or event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(SessionsRevokedEvent, self).__init__(self.NAME, uid, timestamp)


class AppPasswordsRevokedEvent(UserEvent):
    NAME = 'account.app_passwords_revoked'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'info.app_passwords_revoked':
            return cls(event.get('uid'), event.get('value') or event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(AppPasswordsRevokedEvent, self).__init__(self.NAME, uid, timestamp)


class AccountUnsubscribeEvent(UserEvent):
    NAME = 'sid.rm'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event, expected_sids=None):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'sid.rm':
            sid = int(event['value'].split('|')[0])
            if not expected_sids or sid in expected_sids:
                return cls(event.get('uid'), event.get('timestamp'), sid)

    def __init__(self, uid, timestamp, sid):
        super(AccountUnsubscribeEvent, self).__init__(self.NAME, uid, timestamp)
        self.sid = sid
        self.sids = [sid]


class AccountKarmaEvent(UserEvent):
    NAME = 'karma.changed'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def parse_karma_full(cls, karma_full):
        karma_full = karma_full.zfill(4)
        return str(int(karma_full[:-3])), str(int(karma_full[-3:]))

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'info.karma_full':
            karma_status, karma = cls.parse_karma_full(event.get('value', '0'))
            return cls(event.get('uid'), event.get('timestamp'), karma_status, karma)

    def __init__(self, uid, timestamp, karma_status, karma):
        super(AccountKarmaEvent, self).__init__(self.NAME, uid, timestamp)
        self.karma_status = karma_status
        self.karma = karma

    def to_dict(self):
        result = super(AccountKarmaEvent, self).to_dict()
        result['karma_status'] = self.karma_status
        result['karma'] = self.karma
        return result


class SessionKillEvent(UserEvent):
    # событие выхода из сессии
    NAME = 'session.kill'
    PARSER_TYPE = ParserType.AUTH

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('status') == 'ses_kill':
            comment = event.get('comment', '')  # example ssl=1;aid=1409658335000:ea6MJQ:84;ttl=5
            try:
                auth_part = dict(x.split('=', 1) for x in comment.split(';'))
            except ValueError:
                log.warning('Bad comment line: %s', comment)
                return None
            return cls(event.get('uid'), event.get('timestamp'), auth_part.get('aid'))
        else:
            return None

    def __init__(self, uid, timestamp, authid):
        super(SessionKillEvent, self).__init__(self.NAME, uid, timestamp)
        self.authid = authid

    def to_dict(self):
        result = super(SessionKillEvent, self).to_dict()
        result['authid'] = self.authid
        return result


class TokenInvalidateEvent(UserEvent):
    # событие отзыва одного токена
    NAME = 'token.invalidate'
    PARSER_TYPE = ParserType.OAUTH

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('action') == 'invalidate' and
                event.get('target') == 'token'):
            return cls(
                int(event.get('uid')),
                event.get('timestamp'),
                event.get('token_id'),
                event.get('client_id'),
            )
        else:
            return None

    def __init__(self, uid, timestamp, token_id, client_id):
        super(TokenInvalidateEvent, self).__init__(self.NAME, uid, timestamp)
        self.token_id = token_id
        self.client_id = client_id

    def to_dict(self):
        result = super(TokenInvalidateEvent, self).to_dict()
        result['token_id'] = self.token_id
        result['client_id'] = self.client_id
        return result


class TokenInvalidateAllEvent(UserEvent):
    # событие отзыва всех токенов аккаунта
    NAME = 'token.invalidate_all'
    PARSER_TYPE = ParserType.OAUTH

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (parser_type == cls.PARSER_TYPE and
                event.get('action') == 'invalidate' and
                event.get('target') == 'user'):
            return cls(int(event.get('uid')), event.get('timestamp'))
        else:
            return None

    def __init__(self, uid, timestamp):
        super(TokenInvalidateAllEvent, self).__init__(self.NAME, uid, timestamp)


class AuthChallengeEvent(UserEvent):
    # запись окружения пользователя в auth_challenge.log
    NAME = 'account.auth_challenge'
    PARSER_TYPE = ParserType.AUTH_CHALLENGE

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE:
            return cls(
                event.get('uid'),
                event.get('timestamp'),
                event.get('env_profile_id'),
                event.get('env_json'),
                event.get('comment'),
            )

    def __init__(self, uid, timestamp, env_id, env_json, comment):
        self.env_id = env_id
        self.env = env_json
        self.comment = comment
        super(AuthChallengeEvent, self).__init__(self.NAME, uid, timestamp)

    def to_dict(self):
        result = super(AuthChallengeEvent, self).to_dict()
        result['env_id'] = self.env_id
        result['env'] = self.env
        result['comment'] = self.comment
        return result


class AccountDefaultAvatarEvent(UserEvent):
    # изменение или удаление дефолтной аватарки
    NAME = 'account.default_avatar_changed'
    PARSER_TYPE = ParserType.EVENT

    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE and event.get('name') == 'info.default_avatar':
            return cls(event.get('uid'), event.get('timestamp'))

    def __init__(self, uid, timestamp):
        super(AccountDefaultAvatarEvent, self).__init__(self.NAME, uid, timestamp)


class AccountFieldSetToValueEvent(UserEvent):
    PARSER_TYPE = ParserType.EVENT

    # Базовый класс для событий изменения значения поля
    @classmethod
    def is_this_event(cls, parser_type, event):
        if parser_type == cls.PARSER_TYPE:
            if event.get('name') == cls.FIELD_NAME:
                if event.get('value') == cls.FIELD_VALUE:
                    return cls(event.get('uid'), event.get('timestamp'))

    def __init__(self, uid, timestamp):
        if hasattr(self, 'NAME'):
            name = self.NAME
        else:
            name = 'account.%s_%s' % (self.FIELD_NAME, self.FIELD_VALUE)
        super(AccountFieldSetToValueEvent, self).__init__(name, uid, timestamp)
