# -*- coding: utf-8 -*-
import json
import logging
import os

from passport.backend.core.builders.xunistater import get_xunistater
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.backend.core.utils.decorators import cached_property
import six


class BaseHandler(object):
    handler_name = None

    log_metrics = False
    push_metrics_to_xunistater = False
    log_metric_prefix = 'metric:'

    def __init__(self, config, metrics_logger=None, log_metrics=None, push_metrics_to_xunistater=None):
        self.config = config
        self.metrics_logger = metrics_logger or HandlerMetricsLogger()
        self.handler_name = self.__class__.__name__ if self.handler_name is None else self.handler_name

        if log_metrics is not None:
            self.log_metrics = log_metrics
        if push_metrics_to_xunistater is not None:
            self.push_metrics_to_xunistater = push_metrics_to_xunistater

        self._xunistater = None

    @property
    def xunistater(self):
        if self._xunistater is None:
            self._xunistater = get_xunistater()
        return self._xunistater

    def process(self, header, data):
        raise NotImplementedError()

    def flush(self, force=False):
        return True

    def monitor(self, policy):
        """
        Проверка для мониторинга. Возвращает признак окончания проверки, статус мониторинга для отображения
        и сообщение мониторинга (актуальны, если выставлен признак окончания проверки)
        """
        return False, None, None

    def get_message_entries(self, message, **parse_kwargs):
        entries = list(self.parse_message(message, **parse_kwargs))
        if self.log_metrics:
            self.log_message_metrics(message, entries)
        if self.push_metrics_to_xunistater:
            self.push_metrics(message, entries)
        return entries

    def parse_message(self, message, **kwargs):
        raise NotImplementedError()  # Must return list

    def build_metric_name(self, metric_name, log_data, suffix='', server=None):
        return u'{}.{}.{}.{}{}'.format(
            six.u(log_data['handler_name']),
            six.u(metric_name),
            six.u(server or log_data['server']),
            six.u(os.path.normpath(log_data['file']).strip('/')),
            suffix,
        )

    def sanitize_log_metric_name(self, metric_name):
        return metric_name.replace('=', r'\=')

    def log_message_metrics(self, message, entries):
        entries = entries if entries is not None else []
        len_entries = len(entries)

        data = {
            'handler_name': self.handler_name,
            'server': message.server or '_',
            'file': message.file or '_',
        }
        data.update({
            self.sanitize_log_metric_name(
                u'{}{}'.format(
                    six.u(self.log_metric_prefix),
                    self.build_metric_name('entries', data),
                ),
            ): len_entries,
            self.sanitize_log_metric_name(
                u'{}{}'.format(
                    six.u(self.log_metric_prefix),
                    self.build_metric_name('entries', data, server='total'),
                ),
            ): len_entries,
        })
        self.metrics_logger.log(data)

    def push_metrics(self, message, entries):
        entries = entries if entries is not None else []
        data = {
            'handler_name': self.handler_name,
            'server': message.server or '_',
            'file': message.file or '_',
        }
        try:
            len_entries = len(entries)
            self.xunistater.push_metrics({
                self.build_metric_name('entries', data, suffix='_dmmm'): {
                    'value': len_entries,
                },
                self.build_metric_name('entries', data, server='total', suffix='_dmmm'): {
                    'value': len_entries,
                },
            })
        except BaseCoreError as e:
            logging.getLogger('logbroker_client.warning').warning(str(e))
            pass

    @property
    def default_test_header(self):
        return {
            'topic': 'topic1',
            'partition': 1,
            'offset': 1,
            'path': '/var/log/yandex/app1/topic1.log',
            'server': 'server1',
            'ident': 'ident1',
            'logtype': 'tsv',
            'extra_fields': {},
        }

    def parse_test_data_header(self, raw_header):
        header = dict(self.default_test_header)
        if raw_header:
            try:
                header.update(json.loads(raw_header))
            except json.JSONDecodeError as err:
                raise ValueError(
                    'Wrong test data format: HEADER is invalid JSON ()'.format(err),
                )

        return header

    def parse_test_data(self, header, raw_data):
        return raw_data

    def test_data(self, data):
        parts = data.split('|||', 1)
        if len(parts) != 2:
            raise ValueError('Wrong test data format: HEADER|||DATA expected')
        header, data = parts
        data = data.lstrip('\n')
        self.process(
            self.parse_test_data_header(header),
            self.parse_test_data(header, data),
        )


class HandlerMetricsLogger(object):
    def __init__(self, logger=None):
        self._logger = logger

    @property
    def logger(self):
        if self._logger is None:
            self._logger = logging.getLogger('logbroker_client.metrics')
        return self._logger

    def log(self, log_data):
        self.logger.info(dict(
            **log_data
        ))


class BaseHandlerWithRequestId(BaseHandler):
    def parse_message(self, message, **kwargs):
        raise NotImplementedError()

    def process(self, header, data):
        raise NotImplementedError()

    @cached_property
    def request_id_manager(self):
        return RequestIdManager()

    def _reset_prefix(self):
        self.request_id_manager.clear_request_id()

    def _add_prefix(self, prefix):
        self.request_id_manager.push_request_id(prefix)
