# -*- coding: utf-8 -*-
import json
import logging
import time

from google.protobuf import json_format
from google.protobuf.message import DecodeError
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.backend.core.protobuf.logbroker_passport_protocol.logbroker_passport_protocol_pb2 import (
    PassportProtocolMessage,
)
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import ProtocolError
from passport.backend.utils.importlib import import_object
from passport.backend.utils.string import smart_text
import six
import yaml


log_warning = logging.getLogger('logbroker_client.warning')
log = logging.getLogger('logbroker')


class BaseProtobufHandler(BaseHandler):
    MIN_PASSPORT_PROTOCOL_VERSION = (1,)

    def __init__(
        self,
        message_class,
        push_execution_time_to_xunistater=False,
        push_unhandled_exception_to_xunistater=False,
        *args,
        **kwargs
    ):
        super(BaseProtobufHandler, self).__init__(*args, **kwargs)
        self.message_class = import_object(message_class)
        self.message_class_module_name = message_class
        self.push_execution_time_to_xunistater = push_execution_time_to_xunistater
        self.push_unhandled_exception_to_xunistater = push_unhandled_exception_to_xunistater

    def parse_message(self, message, **kwargs):
        raise RuntimeError('Not applicable for protobuf handler class')

    def _parse_message_protobuf(self, data):
        message = self.message_class()
        try:
            message.ParseFromString(data)
        except DecodeError as err:
            raise ProtocolError('Payload proto parse error: {}'.format(err))

        return message

    def _add_log_prefix(self, *values):
        RequestIdManager.push_request_id(*values)

    def _parse_passport_api_meta(self, metadata):
        request_id_string = smart_text(metadata.request_id_string or '-')
        log.debug(u'Got remote request id string: {}'.format(request_id_string))
        self._add_log_prefix(request_id_string)

    def _parse_service_meta(self, decoded, header_type):
        if header_type == 'passport_api':
            self._parse_passport_api_meta(getattr(decoded, header_type))

    def _unpack_passport_pdu(self, protocol_version, data):
        if protocol_version < self.MIN_PASSPORT_PROTOCOL_VERSION:
            raise ProtocolError(
                'Unsupported protocol version {}, minimal is: {}'.format(
                    protocol_version, self.MIN_PASSPORT_PROTOCOL_VERSION,
                ),
            )
        decoded = PassportProtocolMessage()
        try:
            decoded.ParseFromString(data)
        except DecodeError as err:
            raise ProtocolError('Passport proto parse error: {}'.format(err))

        service_meta_type = decoded.header.WhichOneof('service_meta')
        if service_meta_type is not None:
            self._parse_service_meta(decoded.header, service_meta_type)

        return decoded.message_payload

    def _parse_version(self, raw_version):
        try:
            return tuple(int(x) for x in raw_version.split('.'))
        except (TypeError, ValueError) as err:
            raise ProtocolError(
                'Malformed passport protocol version {}: {} {}'.format(
                    raw_version, err.__class__, err,
                ),
            )

    def _parse_pdu(self, header, data):
        raw_version = header.get('passport_protocol_v')
        if raw_version:
            # Паспортный протокол, распаковываем заголовок
            protocol_version = self._parse_version(raw_version)
            message_data = self._unpack_passport_pdu(protocol_version, data)
            return self._parse_message_protobuf(message_data)
        else:
            # Любой другой протокол, распаковываем сразу полученные данные
            return self._parse_message_protobuf(data)

    def process(self, header, data):
        if header.get('message_class') == self.message_class.__name__:
            message = self._parse_pdu(header, data)
            start_ts = time.time()
            exception = False
            try:
                self.process_message(header, message)
            except Exception:
                # Здесь ловятся даже исключения
                # passport.backend.logbroker_client.core.handlers.exceptions.HandlerException,
                # которые класс Worker считает всего лишь варнингами. Я
                # сомневаюсь, что правильно показывать такие исключения в
                # метриках, но поскольку сейчас это работает так, не стану
                # менять поведение.
                exception = True
                raise
            finally:
                if self.log_metrics:
                    self.log_message_metrics(header, data)
                if (
                    self.push_metrics_to_xunistater or
                    self.push_execution_time_to_xunistater or
                    self.push_unhandled_exception_to_xunistater
                ):
                    self.push_metrics_with_time(header, time.time() - start_ts, exception)
                log.debug('Handler executed in {:.3f}s'.format(time.time() - start_ts))

    def _make_metrics_data(self, header):
        return {
            'handler_name': self.handler_name,
            'server': header.get('server'),
            'file': header.get('message_class', '_'),
        }

    def log_message_metrics(self, header, data):
        metrics_data = self._make_metrics_data(header)
        metrics_data.update({
            self.sanitize_log_metric_name(
                u'{}{}'.format(
                    six.u(self.log_metric_prefix),
                    self.build_metric_name('entries', metrics_data),
                ),
            ): 1,
            self.sanitize_log_metric_name(
                u'{}{}'.format(
                    six.u(self.log_metric_prefix),
                    self.build_metric_name('entries', metrics_data, server='total'),
                ),
            ): 1,
        })
        self.metrics_logger.log(metrics_data)

    def _build_metric_records_with_total(self, metrics_data, name, value):
        return {
            self.build_metric_name(name, metrics_data, server='total', suffix='_dmmm'): {
                'value': value,
            },
            self.build_metric_name(name, metrics_data, suffix='_dmmm'): {
                'value': value,
            },
        }

    def push_metrics_with_time(self, header, execution_time, with_exception):
        metrics_data = self._make_metrics_data(header)
        data = {}
        if self.push_metrics_to_xunistater:
            data.update(self._build_metric_records_with_total(
                metrics_data, 'entries', 1,
            ))
        if self.push_execution_time_to_xunistater:
            data.update(self._build_metric_records_with_total(
                metrics_data, 'execution_time', execution_time,
            ))
        if with_exception and self.push_unhandled_exception_to_xunistater:
            data.update(self._build_metric_records_with_total(
                metrics_data, 'unhandled_exception', 1,
            ))
        try:
            self.xunistater.push_metrics(data)
        except BaseCoreError as e:
            log_warning.warning(str(e))

    @property
    def default_test_header(self):
        header = super(BaseProtobufHandler, self).default_test_header
        message_class = self.message_class_module_name.split('.')[-1]
        header['extra_fields'].update(message_class=message_class)
        header.update(
            message_class=message_class,
            path=None,
            logtype=None,
        )
        return header

    def parse_test_data(self, header, raw_data):
        if raw_data.startswith('YAML '):
            try:
                data = yaml.safe_load(raw_data[5:])
            except yaml.YAMLError as err:
                raise ValueError('Wrong test data format: invalid YAML ({})'.format(err))
        else:
            if raw_data.startswith('JSON '):
                raw_data = raw_data[5:]
            try:
                data = json.loads(raw_data)
            except ValueError as err:
                raise ValueError('Wrong test data format: invalid JSON ({})'.format(err))

        try:
            return json_format.ParseDict(data, self.message_class()).SerializeToString()
        except json_format.ParseError as err:
            raise ValueError(
                'Wrong test data format: cannot parse {} as {} ({})'.format(
                    data, self.message_class, err,
                ),
            )

    def process_message(self, header, message):
        """
        Тело обработчика сообщения

        :param header: словарь с данными из хедера:
            - topic
            - partition
            - offset
            - message_class - имя класса protobuf-сообщения
            - extra_fields - словарь дополнительных полей logbroker-сессии
        :param message: десериализованный protobuf-объект сообщения
        """
        raise NotImplementedError()
