# -*- coding: utf-8 -*-
import subprocess
import zlib


class BadArchive(Exception):
    pass


class UnknownArchiveType(BadArchive):
    pass


class BadLzoArchive(BadArchive):
    pass


class BadGzipArchive(BadArchive):
    pass


class Lzo(object):
    def decompress(self, body):
        lzop = subprocess.Popen(
            ('lzop', '-d', '-c'),
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE,
        )
        out, err = lzop.communicate(input=body)
        if err:
            raise BadLzoArchive()
        return out


class Gzip(object):
    WBITS = zlib.MAX_WBITS + 16

    def decompress(self, body):
        try:
            return zlib.decompress(body, self.WBITS)
        except Exception as e:
            raise BadGzipArchive(str(e))


class PlainTSKV(object):

    def decompress(self, body):
        return body


class AutoDecompressor(object):
    LZO_MAGIC = b'\x89LZO'
    GZIP_MAGIC = b'\x1f\x8b'
    PLAIN_TSKV_MAGIC = b'tskv\t'

    def __init__(self):
        self.lzo_decompressor = Lzo()
        self.gzip_decompressor = Gzip()
        self.plain_tskv_decompressor = PlainTSKV()

        self.magic_mapping = [
            (self.LZO_MAGIC, self.lzo_decompressor, None),
            (self.GZIP_MAGIC, self.gzip_decompressor, None),
            (self.PLAIN_TSKV_MAGIC, self.plain_tskv_decompressor, lambda body: body.lstrip()),
        ]

    def decompress(self, body):
        for magic, decompressor, preprocess in self.magic_mapping:
            if preprocess:
                cbody = preprocess(body)
            else:
                cbody = body
            if cbody.startswith(magic):
                return decompressor.decompress(body)
        raise UnknownArchiveType('First bytes: %s', repr(body[:10]))
