""" Config processor for native runner """
import re
import socket
from typing import (
    Any,
    Optional,
)

from dataclasses import dataclass


TConfDict = dict[str, Any]


class ConfigError(ValueError):
    pass


@dataclass
class Config:
    partitions: int
    credentials_config: dict
    client_id: str
    host: str
    topic: str
    handler_class: str
    handler_args: dict
    data_port: int = 2135
    connect_timeout: float = 0.3
    fixed_partitions: bool = False
    ca_cert: Optional[bytes] = None
    decompress: bool = False
    max_count: Optional[int] = None


def get_host_config(per_host_config: TConfDict) -> TConfDict:
    for host_pattern, config in per_host_config.items():
        if re.search(host_pattern, socket.gethostname()):
            return config
    else:
        raise ConfigError('No config for host {}'.format(socket.gethostname()))


def update(conf: TConfDict, values: TConfDict):
    keys = list(values.keys())
    for k in keys:
        if k.startswith('__'):
            _k = k[2:]
            if not isinstance(conf.get(_k), dict):
                raise ConfigError(
                    'Update {} in {}: existing dict element expected'.format(
                        _k, conf,
                    ),
                )
            conf[k[2:]].update(values[k])
            del values[k]
    conf.update(values)


def get_target_config(config: TConfDict, target: TConfDict) -> TConfDict:
    res = dict(config)
    if 'targets' in res:
        del res['targets']
    update(res, target)
    return res


def preprocess_config(config: TConfDict) -> TConfDict:
    config = dict(config)
    if 'ca_cert_file' in config:
        with open(config['ca_cert_file']) as f:
            config['ca_cert'] = f.read()
        del config['ca_cert_file']
    if 'ca_cert' in config:
        config['ca_cert'] = config['ca_cert'].encode()

    return config


def parse_config(config: TConfDict) -> list[Config]:
    res = dict(config['base'])
    if 'per_host' in config:
        update(res, get_host_config(config['per_host']))
    return [
        Config(**preprocess_config(get_target_config(res, target)))
        for target in res['targets']
    ]
