# -*- coding: utf-8 -*-
import argparse
import json
import logging
import logging.config
import socket
import sys

from passport.backend.core.utils.decorators import cached_property
from passport.backend.library.commander import Commander
from passport.backend.library.configurator import Configurator
from passport.backend.logbroker_client.core.consumers.simple.emitter import PartitionsCountEmitter
from passport.backend.logbroker_client.core.logbroker.client import LogbrokerMeta
from passport.backend.logbroker_client.core.monitoring import (
    MONITORING_STATUS_CRIT,
    MONITORING_STATUS_OK,
)
from passport.backend.logbroker_client.core.runner.arbiter import Arbiter
from passport.backend.logbroker_client.core.utils import importobj
import six


log = logging.getLogger(__name__)
DEFAULT_MAX_CLIENT_QUEUE_SIZE = 60000
TEST_MESSAGE_DELIMITER = '----'


class Runner(object):
    def __init__(self, configs, with_passport_settings=False, extra_exported_configs=None):
        self.configs = configs
        self.with_passport_settings = with_passport_settings
        self.extra_exported_configs = extra_exported_configs

    def get_configured_offsets_info(self):
        emitter_args = self.config['emitter'].get('args', {})
        emitter = PartitionsCountEmitter(**emitter_args)
        meta = LogbrokerMeta(emitter.balancer_host, emitter.client)

        partitions = []
        for ident in emitter.idents:
            ident_partitions = meta.show_parts(ident=ident)
            partitions.extend(ident_partitions)
        filtered_topics = emitter.filter_topics(partitions)
        return meta.get_offsets_info(emitter.client, filtered_topics, emitter.data_port, dc=emitter.dc)

    def instantiate_handler(self):
        handler_conf = self.config['workers']['args']['handler']
        handler_cls = importobj(handler_conf.get('class'))
        handler_args = handler_conf.get('args')
        return handler_cls(config=self.config, **handler_args)

    def start_workers(self):
        if self.config.get('use_native_runner'):
            if six.PY2:
                raise RuntimeError('Native runner is py3-only')
            # todo: заменить на обычный импорт после переезда на py3-only
            importobj(
                'passport.backend.logbroker_client.core.native_runner.'
                'runner.NativeRunner',
            )(self.config).run()
        else:
            Arbiter().run(self.config)

    def check_client(self):
        policy = self.config.get('policy', {})

        handler = self.instantiate_handler()
        finalize, code, message = handler.monitor(policy)
        if finalize:
            return code, message

        try:
            offsets_info = self.get_configured_offsets_info()
        except Exception as e:
            log.error('Failed to get queue size', exc_info=e)
            return MONITORING_STATUS_CRIT, 'failed to get queue size'

        max_queue_size = policy.get('max_client_queue_size', DEFAULT_MAX_CLIENT_QUEUE_SIZE)
        queue_size = sum(info['lag'] for info in offsets_info)
        if queue_size > max_queue_size:
            return MONITORING_STATUS_CRIT, 'queue size is %s' % queue_size

        return MONITORING_STATUS_OK, 'OK'

    @staticmethod
    def patch_config_export(config, extra_exported_configs, basename_to_export):
        export_dict = getattr(config, '_by_basenames').get(basename_to_export)
        export_dict.update(extra_exported_configs)
        getattr(config, '_by_basenames')[basename_to_export] = export_dict

    @staticmethod
    def set_debug_loggers(loggers):
        console_handler = logging.StreamHandler(stream=sys.stdout)
        console_handler.setLevel('DEBUG')
        console_handler.setFormatter(logging.Formatter(
            '%(process)d %(asctime)s %(name)-15s %(levelname)-10s %(message)s',
        ))
        if loggers == '*':
            loggers = [None]
        for logger_name in loggers:
            logger = logging.getLogger(logger_name)
            logger.setLevel('DEBUG')
            logger.addHandler(console_handler)

    @cached_property
    def config(self):
        config = Configurator(
            name='logbroker-client',
            configs=self.configs,
        )
        if self.with_passport_settings:
            if self.extra_exported_configs:
                self.patch_config_export(config, self.extra_exported_configs, 'export.yaml')
            config.set_as_passport_settings(basename_to_export='export.yaml')
        config.set_logging()
        if 'logging_update' in config:
            logging.config.dictConfig(config['logging_update'])
        if 'debug_loggers' in config:
            self.set_debug_loggers(config['debug_loggers'])
        return config

    def run(self, monitor):
        if monitor:
            code, message = self.check_client()
            log.debug('monitoring result: %s;%s', code, message)
            print('%d;%s' % (code, message))
        else:
            self.start_workers()

    @staticmethod
    def _in_data_generator(lines_iterator):
        buf = ''
        for line in lines_iterator:
            if line.rstrip('\n') == TEST_MESSAGE_DELIMITER:
                yield buf
                buf = ''
            else:
                buf += line
        if buf:
            yield buf

    def run_test_mode(self, in_file, data):
        if data:
            data_iter = [data]
        else:
            if not in_file:
                in_file = sys.stdin
            data_iter = self._in_data_generator(in_file)

        handler = self.instantiate_handler()
        for data in data_iter:
            handler.test_data(data)

    def _exit_with_heartbeat_error(self, error, batch_mode, crit=True):
        if batch_mode:
            level = '2;crit' if crit else '1;warn'
            print('{} {}'.format(level, error))
        else:
            level = 'Critical' if crit else 'Warning'
            print('{}: {}'.format(level, error))
        exit(255)

    def _print_heartbeat_result(self, result, batch_mode, warn_time, crit_time):
        if not result:
            self._exit_with_heartbeat_error('Empty result', batch_mode=batch_mode)
        max_heartbeat = max(r['last_heartbeat_ago_s'] for r in result.values())
        warn = False
        crit = False
        if 0 < warn_time <= max_heartbeat:
            warn = True
        if 0 < crit_time <= max_heartbeat:
            crit = True

        if not batch_mode:
            print('Max heartbeat: {}'.format(max_heartbeat))
            print('Workers:')
            for pid, worker_data in result.items():
                print('- Worker {}'.format(pid))
                for k in sorted(worker_data.keys()):
                    print('    {}: {}'.format(k, worker_data[k]))
                print()

        if warn or crit:
            self._exit_with_heartbeat_error(
                'Heartbeat is {}'.format(max_heartbeat),
                batch_mode=batch_mode,
                crit=crit,
            )
        elif batch_mode:
            print('0;OK')

    def run_heartbeat_mode(self, request_timeout, batch_mode, warn_time, crit_time):
        stat_socket = self.config.get('stat_socket')
        if not stat_socket:
            raise RuntimeError('No stat_socket option in config')
        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        sock.settimeout(request_timeout)
        try:
            sock.connect(stat_socket)
            sock.sendall(b'HEARTBEAT')
            data = json.loads(sock.recv(102400))
            if 'error' in data:
                six.print_(data['error'], file=sys.stderr)
            else:
                self._print_heartbeat_result(
                    result=data['result'],
                    batch_mode=batch_mode,
                    warn_time=warn_time,
                    crit_time=crit_time,
                )
        except socket.error as e:
            self._exit_with_heartbeat_error(str(e), batch_mode)

    def run_app(self):
        commander = Commander(
            name='command',
            processor=self.run,
        )

        if six.PY3:
            # py2 не поддерживает опциональные сабпарсеры
            test_parser = commander.add_command(
                'test-run',
                required=False,
                processor=self.run_test_mode,
            )
            test_data_group = test_parser.add_mutually_exclusive_group()
            test_data_group.add_argument(
                '-i',
                '--in-file',
                type=argparse.FileType('r'),
                required=False,
                dest='in_file',
            )
            test_data_group.add_argument(
                '-d',
                '--data',
                type=str,
                required=False,
                dest='data',
            )
            heartbeat_parser = commander.add_command(
                'heartbeat',
                required=False,
                processor=self.run_heartbeat_mode,
            )
            heartbeat_parser.add_argument(
                '-t',
                '--request-timeout',
                type=float,
                default=10.0,
                dest='request_timeout',
            )
            heartbeat_parser.add_argument(
                '-w',
                '--warn-time',
                type=float,
                default=-1.0,
                dest='warn_time',
            )
            heartbeat_parser.add_argument(
                '-c',
                '--crit-time',
                type=float,
                default=-1.0,
                dest='crit_time',
            )
            heartbeat_parser.add_argument(
                '-b',
                '--batch-mode',
                action='store_true',
                dest='batch_mode',
            )

        commander.add_argument(
            '-m', '--monitor',
            action='store_true',
            default=False,
            dest='monitor',
        )
        commander.invoke()


def run_app(configs, with_passport_settings=False, extra_exported_configs=None):
    Runner(configs, with_passport_settings, extra_exported_configs).run_app()
