#!/usr/bin/env python
# -*- coding: utf-8 -*-
# python2.6
from datetime import datetime
from optparse import OptionParser
import os
import re
import subprocess
from subprocess import (
    CalledProcessError,
    PIPE,
    Popen,
)
import sys
import time

import six


MONITORING_STATUS_OK = 0
MONITORING_STATUS_WARN = 1
MONITORING_STATUS_CRIT = 2


LINE_RE = re.compile('\[(\w+ \d+ \d+:\d+:\d+\.\d+)\] ([^ ]+) \[\d+\.?\d*]:(\s+)(.*)')


def check_output(*popenargs, **kwargs):
    if 'stdout' in kwargs:
        raise ValueError('stdout argument not allowed, it will be overridden.')
    process = Popen(stdout=PIPE, *popenargs, **kwargs)
    output, unused_err = process.communicate()
    retcode = process.poll()
    if retcode:
        cmd = kwargs.get("args")
        if cmd is None:
            cmd = popenargs[0]
        raise CalledProcessError(retcode, cmd, output=output)
    return output


class ParseError(Exception):
    def __init__(self, output):
        self.push_client_output = output


def print_monitoring(status, text):
    print('%s;%s' % (status, text))


def parse_datetime_to_unixtime(datetime_string):
    dt = datetime.strptime(datetime_string, '%b %d %H:%M:%S.%f')
    dt = dt.replace(year=datetime.today().year)
    return time.mktime(dt.now().timetuple())


def parse_event_datetime_to_unixtime(datetime_string):
    dt = datetime.strptime(datetime_string, '%d.%m.%Y-%H.%M.%S')
    return time.mktime(dt.now().timetuple())


def parse_file_params(key, value):
    handlers = {
        'last_send_time': lambda v: long(v.split(' ')[0]),
        'last_commit_time': lambda v: long(v.split(' ')[0]),
    }
    return handlers.get(key, lambda x: x)(value)


def parse_push_client_status(status_output):
    blocks = []
    current_block = None
    for line in status_output.split('\n')[2:]:
        matched = LINE_RE.match(line)
        if not matched:
            continue
        dt, host, indent_str, body = matched.groups()
        indent = len(indent_str)
        if indent == 2:
            if current_block:
                blocks.append(current_block)
            current_block = {'file': body[1:-3]}
        elif indent == 4:
            key, value = body.split(':', 1)
            key = key.strip()
            value = value.strip()
            current_block[key] = parse_file_params(key, value)
    if current_block:
        blocks.append(current_block)
    return blocks


def check_delivery(push_client_config, delivery_lag_seconds=600):
    output = check_output(
        [
            'push-client',
            '-c', push_client_config,
            '--status',
        ],
        stderr=subprocess.STDOUT,
    )
    try:
        blocks = parse_push_client_status(output)
    except:
        six.reraise(ParseError, ParseError(output))

    silent_files = []
    lags = []
    for block in blocks:
        filepath = block['file']
        last_commit_time = block['last_commit_time']
        modify_time = os.stat(filepath).st_mtime

        lags.append(modify_time - last_commit_time)
        if last_commit_time < modify_time and modify_time - last_commit_time > delivery_lag_seconds:
            silent_files.append(os.path.basename(filepath))

    max_lag = int(max(lags))
    if silent_files:
        files = ','.join(sorted(silent_files))
        exit_code = MONITORING_STATUS_CRIT
        msg = '%s max_lag=%ss' % (files, max_lag)
    else:
        exit_code = MONITORING_STATUS_OK
        msg = 'max_lag=%ss' % max_lag
    return exit_code, msg


def main():
    parser = OptionParser()
    parser.add_option(
        '-c', '--config', dest='config', action='store',
        help='push-client config',
    )

    parser.add_option(
        '-l', '--lag-size', dest='lag_size', action='store', type='int', default=600,
        help='allowed lag size in seconds (default: 600s)',
    )

    parser.add_option(
        '-t', '--show-trace',
        action='store_true', dest='show_trace', default=False,
        help='print traceback on error',
    )

    (options, args) = parser.parse_args()
    config = options.config
    exit_code = 0
    if not config:
        exit_code = MONITORING_STATUS_CRIT
        print_monitoring(exit_code, 'missing `config` argument')
        return exit_code

    lag_size = options.lag_size
    show_trace = options.show_trace

    try:
        try:
            exit_code, msg = check_delivery(config, lag_size)
            print_monitoring(exit_code, msg)
            return exit_code
        except subprocess.CalledProcessError as e:
            exit_code = MONITORING_STATUS_CRIT
            print_monitoring(exit_code, str(e))
            raise
        except ParseError as e:
            exit_code = MONITORING_STATUS_CRIT
            print_monitoring(exit_code, 'parse_error: %s' % e.push_client_output[:50].replace('\n', '\\n'))
            raise
    except:
        if show_trace:
            raise
        return exit_code


if __name__ == '__main__':
    sys.exit(main())
