# -*- coding: utf-8 -*-
from contextlib import contextmanager
from functools import wraps
from importlib import import_module
import logging
import os
import random
import time
import uuid

from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.backend.core.types.account.account import (
    PDD_UID_BOUNDARY,
    YATEAM_UID_BOUNDARY,
)


log = logging.getLogger(__name__)


class LogPrefixManager(object):
    system_state = None

    @staticmethod
    def clear_log_prefix():
        RequestIdManager.clear_request_id()

    @staticmethod
    def add_log_prefix(*values):
        RequestIdManager.push_request_id(*values)

    @classmethod
    @contextmanager
    def system_context(cls, *values):
        cls.system_state = values
        cls.clear_log_prefix()
        try:
            cls.add_log_prefix(*values)
            yield
        finally:
            cls.system_state = None
            cls.clear_log_prefix()

    @classmethod
    def recover_system_context(cls):
        if cls.system_state:
            cls.add_log_prefix(*cls.system_state)

    @classmethod
    @contextmanager
    def user_context(cls, *values):
        cls.clear_log_prefix()
        cls.recover_system_context()
        try:
            cls.add_log_prefix(*values)
            yield
        finally:
            cls.clear_log_prefix()
            cls.recover_system_context()

    @staticmethod
    def new_id():
        return str(uuid.uuid4())[-12:]


def find(matcher, top='.'):
    for dirpath, _dirnames, filenames in os.walk(top):
        for f in filenames:
            f = os.path.relpath(os.path.join(dirpath, f), top)
            basename = os.path.basename(f)
            if matcher.match(basename):
                yield f


def importobj(objpath):
    module_path, obj_name = objpath.rsplit('.', 1)
    try:
        module = import_module(module_path)
    except ImportError as e:
        raise ImportError(objpath, str(e))
    return getattr(module, obj_name)


def retriable_n(retry_count=3, time_sleep=0.2, exceptions=(Exception,)):
    def retriable_n_deco(func):
        @wraps(func)
        def wrapper(*args, **kw):
            for i in range(retry_count - 1):
                try:
                    return func(*args, **kw)
                except Exception as e:
                    if isinstance(e, exceptions):
                        log.warning('%s(*%s, **%s) try %i failed, retrying: %s', func.__name__, args, kw, i, e)
                        time.sleep(time_sleep)
                    else:
                        raise
            else:
                return func(*args, **kw)

        return wrapper

    return retriable_n_deco


class ExponentialBackoff(object):
    def __init__(self, init_value=0.5, max_value=60 * 5, factor=2, jitter=0.3):
        self.init_value = init_value
        self.current_backoff = None
        self.max_value = max_value
        self.factor = factor
        self.jitter = jitter

    def reset(self):
        self.current_backoff = None

    def is_active(self):
        return self.current_backoff is not None

    def get(self):
        return float(self.current_backoff) * (1 + self.jitter * random.random())

    def update(self):
        if self.current_backoff is None:
            self.current_backoff = self.init_value
        else:
            self.current_backoff = min(self.current_backoff * self.factor, self.max_value)


def is_yateam_uid(uid):
    return YATEAM_UID_BOUNDARY <= int(uid) < PDD_UID_BOUNDARY
