# -*- coding: utf-8 -*-
import logging

from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.library.historydbloader.historydb.parser import ParserType
from passport.backend.logbroker_client.core.events.events import UserEvent


log = logging.getLogger(__name__)


class ValidationError(Exception):
    pass


class BaseAccountEntityChangeEvent(UserEvent):
    NAME = ''
    ENTITY = ''
    OPERATIONS = [
        'added',
        'updated',
        'created',
    ]

    PARSER_TYPE = ParserType.PASSPORT_STATBOX

    @classmethod
    def is_this_event(cls, parser_type, event):
        if (
            parser_type == cls.PARSER_TYPE
            and event.get('entity') == cls.ENTITY
            and event.get('operation') in cls.OPERATIONS
            and 'uid' in event
            and 'unixtime' in event
        ):
            try:
                cls.validate_event_fields(event)
            except ValidationError as err:
                log.warning(
                    '{} event validation failed: {} "{}"'.format(
                        cls.__name__, err, trim_message(str(event)),
                    ),
                )
                return None
            return cls(event=event, uid=event['uid'], timestamp=event['unixtime'])
        else:
            return None

    @classmethod
    def validate_event_fields(cls, event):
        return

    def __init__(self, event, uid, timestamp):
        self.event = event
        self.operation = event.get('operation')
        super(BaseAccountEntityChangeEvent, self).__init__(
            name=self.NAME,
            uid=uid,
            timestamp=timestamp,
        )

    def __str__(self):
        return '<{} {}>'.format(self.__class__.__name__, self.operation)


class UnsubscribedFromMaillistsAttributeChangeEvent(BaseAccountEntityChangeEvent):
    NAME = 'unsubscribed_from_maillists_change'
    ENTITY = 'account.unsubscribed_from_maillists'


class Sid2ChangeEvent(BaseAccountEntityChangeEvent):
    NAME = 'sid2_change'
    ENTITY = 'subscriptions'
    OPERATIONS = ['added', 'removed']

    def __init__(self, event, uid, timestamp):
        super(Sid2ChangeEvent, self).__init__(event=event, uid=uid, timestamp=timestamp)

    @classmethod
    def is_this_event(cls, parser_type, event):
        if event.get('sid') != '2':
            return None
        return super(Sid2ChangeEvent, cls).is_this_event(parser_type, event)


class EmailConfirmedAddEvent(BaseAccountEntityChangeEvent):
    NAME = 'email_add_confirmed'
    ENTITY = 'account.emails'
    OPERATIONS = ['added']

    @classmethod
    def is_this_event(cls, parser_type, event):
        if not event.get('confirmed_at'):
            return None
        return super(EmailConfirmedAddEvent, cls).is_this_event(parser_type, event)

    @classmethod
    def validate_event_fields(cls, event):
        try:
            int(event.get('email_id'))
        except ValueError:
            raise ValidationError('Wrong or missing email_id')

    def __init__(self, event, uid, timestamp):
        self.email_id = int(event.get('email_id'))
        super(EmailConfirmedAddEvent, self).__init__(event=event, uid=uid, timestamp=timestamp)


class PortalAliasAddEvent(BaseAccountEntityChangeEvent):
    NAME = 'portal_alias_add'
    ENTITY = 'aliases'
    OPERATIONS = ['added']

    @classmethod
    def is_this_event(cls, parser_type, event):
        if event.get('type') != '1':
            return None
        return super(PortalAliasAddEvent, cls).is_this_event(parser_type, event)
