# -*- coding: utf-8 -*-
from collections import defaultdict
from concurrent.futures import ThreadPoolExecutor
from functools import partial
import logging
import time
import uuid

from passport.backend.core.builders.blackbox import (
    Blackbox,
    BlackboxTemporaryError,
)
from passport.backend.core.builders.sender_api.exceptions import SenderApiTemporaryError
from passport.backend.core.builders.sender_api.sender_api import SenderApi
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.mail_subscriptions.services_manager import MailSubscriptionServicesManager
from passport.backend.core.models.account import Account
from passport.backend.core.models.email import Email
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.core.undefined import Undefined
from passport.backend.core.utils.decorators import cached_property
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandlerWithRequestId
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.mail_unsubscriptions.events import (
    EmailConfirmedAddEvent,
    PortalAliasAddEvent,
    Sid2ChangeEvent,
    UnsubscribedFromMaillistsAttributeChangeEvent,
)
from passport.backend.logbroker_client.mail_unsubscriptions.tools import get_tvm_credentials_manager_synchronized
from passport.backend.utils.string import smart_text
from passport.backend.utils.time import unixtime_to_datetime


log = logging.getLogger('logbroker')


class ServiceHandlerException(HandlerException):
    pass


class DebugMetric:
    def __init__(self, threshold=10.0):
        self.threshold = threshold
        self.last_tick = time.time()
        self.messages_by_topic = defaultdict(int)
        self.events_by_topic = defaultdict(int)
        self.empty_messages_by_topic = defaultdict(int)

    @staticmethod
    def _gen_message(topic, messages, events, empty_messages):
        return '{}: {}'.format(
            topic,
            ' '.join(
                '{}={}'.format(k, v)
                for k, v in dict(
                    messages=messages,
                    events=events,
                    empty_messages=empty_messages,
                ).items()
            )
        )

    def _beacon(self, shift):
        msg = 'METRIC: processed in {:.3f} sec {}'.format(
            shift,
            '; '.join(
                self._gen_message(
                    topic,
                    self.messages_by_topic[topic],
                    self.events_by_topic[topic],
                    self.empty_messages_by_topic[topic],
                )
                for topic in self.messages_by_topic.keys()
            ),
        )
        if len(self.messages_by_topic) > 1:
            msg += ' ' + self._gen_message(
                'Total',
                sum(self.messages_by_topic.values()),
                sum(self.events_by_topic.values()),
                sum(self.empty_messages_by_topic.values()),
            )
        log.debug(msg)

    def _tick(self):
        now = time.time()
        shift = now - self.last_tick
        if shift >= self.threshold:
            self._beacon(shift)
            self.last_tick = now
            self.messages_by_topic = defaultdict(int)
            self.events_by_topic = defaultdict(int)
            self.empty_messages_by_topic = defaultdict(int)

    def message(self, topic, empty=False):
        if empty:
            self.empty_messages_by_topic[topic] += 1
        else:
            self.messages_by_topic[topic] += 1
        self._tick()

    def event(self, topic):
        self.events_by_topic[topic] += 1


class MailUnsubscriptionsHandler(BaseHandlerWithRequestId):
    handler_name = 'mail_unsubscriptions'
    event_classes = [
        UnsubscribedFromMaillistsAttributeChangeEvent,
        Sid2ChangeEvent,
        EmailConfirmedAddEvent,
        PortalAliasAddEvent,
    ]

    def __init__(self, config, thread_count=1, dry_run=False, **kwargs):
        super(MailUnsubscriptionsHandler, self).__init__(config=config, **kwargs)
        self.filter = BasicFilter(self.event_classes)

        self.dry_run = dry_run

        assert thread_count > 0
        self.thread_count = thread_count
        self.executor_pool = None
        if self.thread_count > 1:
            log.debug('Starting executor pool of {} threads'.format(self.thread_count))
            self.executor_pool = ThreadPoolExecutor(max_workers=self.thread_count)

        self.debug_metric = DebugMetric()

    @cached_property
    def sender(self):
        return SenderApi()

    @cached_property
    def blackbox(self):
        return Blackbox(tvm_credentials_manager=get_tvm_credentials_manager_synchronized())

    @cached_property
    def subscriptions(self):
        return MailSubscriptionServicesManager()

    def parse_message(self, message, **_):
        return self.filter.filter(message)

    def _complete_futures_on_error(self, futures):
        for future in futures:
            if not future.done():
                try:
                    future.result()
                except Exception as err:
                    log.error(
                        'Exception {} in future while waiting for it to '
                        'complete in other future\'s exception handler'.format(err),
                    )

    def _process_events_async(self, events, server, topic, processor_fn):
        log.debug('Async jobs for {}/{}: starting {}'.format(server, topic, len(events)))
        futures = [self.executor_pool.submit(processor_fn, event) for event in events]
        try:
            for i, future in enumerate(futures, 1):
                future.result()
                log.debug('Async jobs for {}/{}: ended {}/{}'.format(i, server, topic, len(events)))
        except Exception as err:
            log.error(
                'Exception {} while processing events. '
                'Wait for rest to finish..'.format(err),
            )
            self._complete_futures_on_error(futures)
            raise
        log.debug('Async jobs for {}/{}: completed {}'.format(server, topic, len(events)))

    def process(self, header, data):
        message = MessageChunk(header, smart_text(data, errors='ignore'))
        events = self.get_message_entries(message)
        self.debug_metric.message(message.topic, empty=(len(events) == 0))

        if not events:
            return True

        if self.thread_count > 1:
            self._process_events_async(
                events,
                message.server,
                message.topic,
                partial(self.process_event, server=message.server, topic=message.topic),
            )
        else:
            for event in events:
                self.process_event(event, message.server, message.topic)

        return True

    def process_event(self, event, server, topic):
        self._reset_prefix()
        self._add_prefix(str(uuid.uuid4())[:8])
        self._add_prefix(event.uid)
        log.debug('New event {} server={} topic={} time={:%Y-%m-%d %H:%M:%S}'.format(
            event,
            server,
            topic,
            unixtime_to_datetime(event.timestamp) if event.timestamp else '-',
        ))
        try:
            self._process_event(event)
        except (BlackboxTemporaryError, SenderApiTemporaryError) as e:
            raise ServiceHandlerException(
                '{}.{}: {}'.format(e.__module__, e.__class__.__name__, e),
            )
        except Exception:
            log.exception('Exception in handler')
            raise
        finally:
            self.debug_metric.event(topic)

    def _process_event(self, event):
        if isinstance(event, UnsubscribedFromMaillistsAttributeChangeEvent):
            self.process_attr_unsubscribed_from_maillists_change(event.uid)
        elif isinstance(event, Sid2ChangeEvent):
            self.process_sid2(event.uid)
        elif isinstance(event, EmailConfirmedAddEvent):
            self.process_new_email(event.uid, event.email_id)
        elif isinstance(event, PortalAliasAddEvent):
            self.process_new_portal_alias(event.uid)
        else:
            log.error('Unexpected event {}'.format(event))

    def _get_account(self, uid, **userinfo_kwargs):
        userinfo = self.blackbox.userinfo(
            uid=uid,
            need_aliases=True,
            get_hidden_aliases=False,
            need_display_name=False,
            emails=True,
            attributes=['account.unsubscribed_from_maillists'],
            **userinfo_kwargs
        )
        try:
            account = Account().parse(userinfo)
            return account
        except UnknownUid:
            return None

    def _fold_native_emails(self, emails):
        result = set()
        for email in emails:
            username, domain = Email.split(email)
            if domain in settings.NATIVE_EMAIL_DOMAINS:
                result.add(username + '@' + settings.NATIVE_EMAIL_DOMAINS[0])
            else:
                result.add(email)
        return result

    def _account_to_native_emails(self, account):
        emails = list({
            email.normalized_address
            for email in account.emails.all
            if email.is_native
        })
        emails = self._fold_native_emails(emails)
        if account.is_normal:
            emails.add(account.portal_alias.alias + '@' + settings.NATIVE_EMAIL_DOMAINS[0])
        log.debug('Native emails: {}'.format([
            mask_email_for_statbox(email) for email in emails
        ]))
        return sorted(list(emails))

    def _account_to_non_native_emails(self, account):
        emails = list({
            email.normalized_address
            for email in account.emails.all
            if not email.is_native and email.is_confirmed
        })
        log.debug('Non-native emails: {}'.format([
            mask_email_for_statbox(email) for email in emails
        ]))
        return emails

    def _subscription_state(self, account):
        unsubscribed = []
        subscribed = []
        for service in self.subscriptions.get_all_services():
            if service.id in account.unsubscribed_from_maillists:
                unsubscribed.append(service)
            else:
                subscribed.append(service)

        return unsubscribed, subscribed

    def _dry_run_comment(self):
        return ' (dry run)' if self.dry_run else ''

    def _apply_unsubscriptions_state(self, email, unsubscribe, subscribe):
        log.debug(
            'Applying subscription state{}'
            'email={} unsubscribe={} subscribe={}'.format(
                self._dry_run_comment(), mask_email_for_statbox(email),
                unsubscribe, subscribe,
            ),
        )
        unsubscribe_ext_ids = self._services_to_external_ids(unsubscribe)
        subscribe_ext_ids = self._services_to_external_ids(subscribe)
        if not (unsubscribe_ext_ids or subscribe_ext_ids):
            log.warning('No subscribe_ext_ids nor unsubscribe_ext_ids')
            return
        if self.dry_run:
            return
        self.sender.set_unsubscriptions(email, unsubscribe_ext_ids, subscribe_ext_ids)

    def _copy_unsubscriptions(self, email_src, email_dst):
        log.debug(
            'Copying unsubscriptions{} from {} to {}'.format(
                self._dry_run_comment(),
                mask_email_for_statbox(email_src),
                mask_email_for_statbox(email_dst),
            ),
        )
        if self.dry_run:
            return
        self.sender.copy_unsubscriptions(email_src=email_src, email_dst=email_dst)

    @staticmethod
    def _services_to_external_ids(services):
        if not services:
            return []
        return list(set(sum((s.external_list_ids for s in services), [])))

    def process_attr_unsubscribed_from_maillists_change(self, uid):
        log.debug('unsubscribed_from_maillists changed')
        account = self._get_account(
            uid=uid,
            email_attributes='all',
        )
        if account is None:
            log.warning('Uid {} not found. Skipping'.format(uid))
            return
        self._process_attr_unsubscribed_from_maillists_change(account)

    def _process_attr_unsubscribed_from_maillists_change(self, account):
        """ Attribute unsubscribed_from_maillists change event """
        native_emails = self._account_to_native_emails(account)
        emails = self._account_to_non_native_emails(account)
        unsubscribed, subscribed = self._subscription_state(account)

        log.debug(
            'Applying new value: {}'.format(account.unsubscribed_from_maillists),
        )
        for email in native_emails:
            self._apply_unsubscriptions_state(email, unsubscribed, subscribed)

        if unsubscribed and native_emails:
            native_email = native_emails[0]
            for email in emails:
                self._copy_unsubscriptions(native_email, email)

    def process_sid2(self, uid):
        """ Sid 2 subscription change event """
        log.debug('SID 2 state changed')
        account = self._get_account(
            uid=uid,
            dbfields=[
                'subscription.suid.2',
                'subscription.login_rule.2',
                'subscription.login.2',
                'subscription.host_id.2',
            ],
        )
        if account is None:
            log.warning('Uid {} not found. Skipping'.format(uid))
            return
        subscribed_to_sid2 = account.subscriptions is not Undefined and account.has_sid(2)
        if not account.unsubscribed_from_maillists:
            log.debug('Attr unsubscribed_from_maillists is empty. Skipping')
            return

        log.debug(
            '{} native email unsubscriptions {}'.format(
                'Setting' if subscribed_to_sid2 else 'Cancelling',
                account.unsubscribed_from_maillists,
            ),
        )
        native_emails = self._account_to_native_emails(account)
        current_unsubscriptions, _ = self._subscription_state(account)

        for email in native_emails:
            # Если подписались на 2-й сид - применяем почтовые отписки
            # Если отписались от 2-го сида - отменяем почтовые отписки
            if subscribed_to_sid2:
                unsubscribe = current_unsubscriptions
                subscribe = []
            else:
                subscribe = current_unsubscriptions
                unsubscribe = []
            self._apply_unsubscriptions_state(email, unsubscribe, subscribe)

    def process_new_email(self, uid, email_id):
        log.debug('New email id={}'.format(email_id))
        account = self._get_account(
            uid=uid,
            email_attributes='all',
        )
        if account is None:
            log.warning('Uid {} not found. Skipping'.format(uid))
            return
        if not account.unsubscribed_from_maillists:
            log.debug('Attr unsubscribed_from_maillists is empty. Skipping')
            return
        email_obj = account.emails.get_by_id(email_id)
        if email_obj is None:
            log.warning('Email id={} not found'.format(email_id))
            return
        if not email_obj.is_confirmed:
            log.warning('Email id={} is not confirmed'.format(email_id))
            return
        email = email_obj.normalized_address
        if not account.emails.native:
            log.debug(
                'Account has no native emails to copy unsubscriptions '
                'to non-native one {}'.format(
                    mask_email_for_statbox(email),
                ),
            )
            return
        native_email = account.emails.native[0].normalized_address
        log.debug(
            'Copying unsubscriptions from native email {} to {}'.format(
                mask_email_for_statbox(native_email),
                mask_email_for_statbox(email),
            ))
        self._copy_unsubscriptions(native_email, email)

    def process_new_portal_alias(self, uid):
        log.debug('New portal alias')
        account = self._get_account(
            uid=uid,
        )
        if account is None:
            log.warning('Uid {} not found. Skipping'.format(uid))
            return
        if not account.unsubscribed_from_maillists:
            log.debug('Attr unsubscribed_from_maillists is empty. Skipping')
            return

        native_emails = self._account_to_native_emails(account)
        unsubscribed, subscribed = self._subscription_state(account)

        log.debug(
            'Applying value: {} to native emails {}'.format(
                account.unsubscribed_from_maillists,
                [mask_email_for_statbox(email) for email in native_emails],
            ),
        )
        for email in native_emails:
            self._apply_unsubscriptions_state(email, unsubscribed, subscribed)
