# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.core.builders.mixins.json_parser.json_parser import JsonParserMixin
from passport.backend.logbroker_client.oauth.exceptions import (
    BaseOAuthError,
    OAuthPersistentError,
    OAuthRequestError,
    OAuthTemporaryError,
)


log = logging.getLogger(__name__)


class OAuth(BaseBuilder, JsonParserMixin):
    base_error_class = BaseOAuthError
    temporary_error_class = OAuthTemporaryError
    parser_error_class = OAuthRequestError

    def __init__(self, url, timeout, retries=3, useragent=None, consumer='passport-lbc'):
        self.consumer = consumer
        super(OAuth, self).__init__(
            url=url,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
        )

    def check_response_for_errors(self, data, raw_response):
        if raw_response.status_code == 400:
            raise OAuthPersistentError()
        elif raw_response.status_code != 200:
            raise OAuthTemporaryError()
        elif data.get('status') != 'ok':
            raise OAuthRequestError()

    def refresh_token(self, token_id):
        url = '/api/1/token/refresh'
        data = {
            'token_id': token_id,
        }
        params = {
            'consumer': self.consumer,
        }
        return self._request_with_retries_simple(
            url_suffix=url,
            method='POST',
            error_detector=self.check_response_for_errors,
            parser=self.parse_json,
            params=params,
            data=data,
        )
