# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.avatars_mds_api import get_avatars_mds_api
from passport.backend.core.builders.blackbox.blackbox import Blackbox
from passport.backend.core.builders.blackbox.exceptions import BaseBlackboxError
from passport.backend.core.builders.clean_web_api import (
    CleanWebPermanentError,
    CleanWebTemporaryError,
    get_clean_web_api,
)
from passport.backend.core.builders.passport import (
    Passport,
    PassportPermanentError,
    PassportTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.models.account import (
    Account,
    UnknownUid,
)
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.types.login.login import is_test_yandex_login
from passport.backend.core.utils.decorators import cached_property
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.passport_clean_web.events import (
    PassportNewDefaultAvatarEvent,
    PassportNewDisplaynameEvent,
    PassportNewFullnameEvent,
)


log = logging.getLogger(__name__)


class PassportCleanWebHandlerException(HandlerException):
    pass


class PassportCleanWebHandler(BaseHandler):
    handler_name = 'passport_clean_web'

    def __init__(self, config,  **kwargs):
        super(PassportCleanWebHandler, self).__init__(config=config, **kwargs)
        events_classes = [
            PassportNewDefaultAvatarEvent,
            PassportNewDisplaynameEvent,
            PassportNewFullnameEvent,
        ]
        self.filter = BasicFilter(events_classes)

    def parse_message(self, message):
        return self.filter.filter(message)

    def process(self, header, data):
        message = MessageChunk(header, data)
        events = self.get_message_entries(message)
        for event in events:
            self.process_event(event)
        return True

    @cached_property
    def avatars_mds(self):
        return get_avatars_mds_api('yapic')

    @cached_property
    def blackbox(self):
        return Blackbox(
            settings.BLACKBOX_URL,
            use_tvm=settings.BLACKBOX_USE_TVM,
        )

    @cached_property
    def clean_web_api(self):
        return get_clean_web_api()

    @cached_property
    def passport(self):
        return Passport(
            use_tvm=settings.PASSPORT_USE_TVM,
        )

    def process_event(self, event):
        try:
            userinfo = self.blackbox.userinfo(
                uid=event.uid,
                attributes=[],
                dbfields=[],
                need_display_name=False,
                need_public_name=False,
                need_aliases=True,
            )
            account = Account().parse(userinfo)
            if account.login and is_test_yandex_login(account.login):
                return

            if isinstance(event, PassportNewDefaultAvatarEvent):
                avatar_key = event.avatar_key
                group_id, key = avatar_key.split('/')
                url = self.avatars_mds.get_read_url(group_id, key, 'orig')

                clean_web_auto_response = self.clean_web_api.check_media(
                    url,
                    key='avatar_cl_check_%s_%s' % (event.uid, event.timestamp),
                    simple_response=True,
                    auto_only=settings.CLEAN_WEB_AVATARS_AUTO_ONLY,
                    uid=event.uid,
                    verdict_data=json.dumps(dict(
                        uid=event.uid,
                        avatar_key=avatar_key,
                    )),
                )
                log.debug('clean web responded "{}" for url "{}"'.format(clean_web_auto_response, url))

                if settings.CLEAN_WEB_USE_AUTO_VERDICTS and clean_web_auto_response is False:
                    self.passport.reset_avatar(event.uid, avatar_key)
            elif isinstance(event, (PassportNewFullnameEvent, PassportNewDisplaynameEvent)):
                value_to_send = event.value
                if isinstance(event, PassportNewDisplaynameEvent):
                    display_name = DisplayName()
                    display_name.set(value_to_send)
                    value_to_send = display_name.public_name
                if not settings.CLEAN_WEB_NAMES_AUTO_ONLY:
                    self.clean_web_api.check_user_data(
                        key='{}_cl_check_{}_{}'.format(event.field, event.uid, event.timestamp),
                        auto_only=False,
                        uid=event.uid,
                        verdict_data=json.dumps(dict(
                            uid=event.uid,
                            value=value_to_send,
                        )),
                        **{
                            event.field: value_to_send,
                        }
                    )

        except UnknownUid:
            pass
        except (BaseBlackboxError, CleanWebTemporaryError, PassportTemporaryError) as e:
            log.error('Error while processing new avatar: %s', e)
            raise PassportCleanWebHandlerException(e)
        except (CleanWebPermanentError, PassportPermanentError) as e:
            log.error('Error while processing new avatar (won\'t retry): %s', e)
            return
