# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.passport import (
    Passport,
    PassportAccountNotFoundError,
    PassportActionImpossible,
    PassportPermanentError,
    PassportTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.utils.decorators import cached_property
from passport.backend.library.historydbloader.historydb.parser import ParserType
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.passport_toloka.events import TolokaEvent


log = logging.getLogger(__name__)


class PassportTolokaHandlerException(HandlerException):
    pass


# Нужен этот фильтр, потому что не приходит имя файла лога в хедере
class AdhocFilter(BasicFilter):
    def get_message_parser_type(self, message):
        return ParserType.PASSPORT_TOLOKA


class PassportTolokaHandler(BaseHandler):
    handler_name = 'passport_toloka'

    def __init__(self, config,  **kwargs):
        super(PassportTolokaHandler, self).__init__(config=config, **kwargs)
        events_classes = [TolokaEvent]
        self.filter = AdhocFilter(events_classes)

    def parse_message(self, message):
        return self.filter.filter(message)

    def process(self, header, data):
        message = MessageChunk(header, data)
        events = self.get_message_entries(message)
        for event in events:
            self.process_event(event)
        return True

    @cached_property
    def passport(self):
        return Passport(
            use_tvm=settings.PASSPORT_USE_TVM,
        )

    def process_event(self, event):
        if not event.data:
            log.warning(u'Got verdict "{}" without data. Can not process it.'.format(event))
            return
        if event.resolution not in settings.CLEAN_WEB_REJECTED_CAUSES and event.resolution not in settings.CLEAN_WEB_APPROVED_CAUSES:
            log.warning(u'Got verdict "{}" with unknown cause. Ignoring it.'.format(event))
            return
        try:
            if event.entity == 'image':
                uid = event.data['uid']
                try:
                    avatar_key = event.data['avatar_key']
                except KeyError:
                    log.exception('Wrong event structure')
                    return
                log.debug(u'clean web responded "{}" with cause "{}" for avatar "{}"'.format(event.value, event.resolution, avatar_key))
                if event.resolution in settings.RESOLUTIONS_FOR_AVATAR and event.value is True:
                    log.debug(u'Avatar should be reset {}. uid: {} clean web resolution: {} avatar_key: {}'.format(
                        u'(but it is DRY_RUN)' if settings.DRY_RUN_AVATARS else u'',
                        uid,
                        event.resolution,
                        avatar_key,
                    ))
                    if not settings.DRY_RUN_AVATARS:
                        try:
                            self.passport.reset_avatar(
                                uid,
                                avatar_key,
                            )
                            log.warning(u'Avatar was reset. Uid: {}, clean web resolution: {}, avatar_key: {}.'.format(
                                uid,
                                event.resolution,
                                avatar_key,
                            ))
                        except PassportAccountNotFoundError:
                            log.debug(u'Account with uid {} is not found'.format(uid))

            elif event.entity in ('display_name', 'full_name'):
                uid = event.data['uid']
                value = event.data['value']
                log.debug(
                    u'clean web responded "{response}" with cause "{resolution}" for {entity} "{name}" of uid={uid}'.format(
                        response=event.value,
                        resolution=event.resolution,
                        entity=event.entity,
                        name=value,
                        uid=uid,
                    )
                )
                if event.resolution in settings.RESOLUTIONS_FOR_NAME and event.value is True:
                    log.debug(u'Display name should be reset {}. uid: {} clean web resolution: {} value: {}'.format(
                        u'(but it is DRY_RUN)' if settings.DRY_RUN_DISPLAY_NAME else u'',
                        uid,
                        event.resolution,
                        value,
                    ))
                    if not settings.DRY_RUN_DISPLAY_NAME:
                        try:
                            self.passport.reset_display_name(
                                uid,
                                full_name=value if event.entity == 'full_name' else None,
                                public_name=value if event.entity == 'display_name' else None,
                            )
                            log.warning(u'Display name was reset. Uid: {}, clean web resolution: {}.'.format(
                                uid,
                                event.resolution,
                            ))
                        except PassportAccountNotFoundError:
                            log.debug(u'Account with uid {} is not found'.format(uid))
                        except PassportActionImpossible:
                            log.debug(u'Account with uid {} has the different name from the name transmitted in the request'.format(uid))
            else:
                log.warning(u'Got verdict for entity "{}". Can not process it'.format(event.entity))

        except PassportTemporaryError as e:
            log.error(u'Error while processing toloka verdict: %s', e)
            raise PassportTolokaHandlerException(e)
        except PassportPermanentError as e:
            log.error(u'Error while processing toloka verdict (won\'t retry): %s', e)
            return
