# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.oauth import (
    OAuth,
    OAuthAccountNotFoundError,
    BaseOAuthError,
)
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.revoke_oauth_tokens.events import RevokeOAuthTokensEvent


log = logging.getLogger(__name__)


class RevokeOAuthTokensHandlerException(HandlerException):
    pass


class RevokeOAuthTokensHandler(BaseHandler):
    handler_name = 'revoke_oauth_tokens'

    def __init__(self, config, **kwargs):
        super(RevokeOAuthTokensHandler, self).__init__(config=config, **kwargs)
        events_classes = [RevokeOAuthTokensEvent]
        self.filter = BasicFilter(events_classes)
        self.oauth = OAuth(tvm_dst_alias=None)

    def parse_message(self, message):
        return self.filter.filter(message)

    def process(self, header, data):
        message = MessageChunk(header, data)
        events = self.get_message_entries(message)
        for event in events:
            self.process_event(event)
        return True

    def process_event(self, event):
        uid = event.uid
        device_id = event.device_id
        user_ip = event.user_ip
        try:
            self.oauth.revoke_device(uid, device_id, user_ip)
        except OAuthAccountNotFoundError:
            log.debug('Account %s not found.' % (uid,))
        except BaseOAuthError as e:
            raise RevokeOAuthTokensHandlerException('Coundn\'t revoke oauth tokens: %s' % e)
        else:
            log.debug(
                'Revoked oauth tokens on device. Uid=%s, device_id=%s, user_ip=%s',
                uid,
                device_id,
                user_ip,
            )
