# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders.social_broker.exceptions import (
    BaseSocialBrokerError,
    SocialBrokerProfileNotAllowedError,
)
from passport.backend.core.builders.social_broker.social_broker import SocialBroker
from passport.backend.core.builders.taxi_zalogin import (
    BaseTaxiZaloginError,
    TaxiZalogin,
)
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.social_bindings.events import (
    BindPhonishAccountByTrackEvent,
    SocialBindingsEvent,
)


log = logging.getLogger(__name__)


class SocialBindingsHandlerException(HandlerException):
    pass


class SocialBindingsHandler(BaseHandler):
    handler_name = 'social_bindings'

    def __init__(self, config, **kwargs):
        super(SocialBindingsHandler, self).__init__(config=config, **kwargs)
        events_classes = [
            BindPhonishAccountByTrackEvent,
            SocialBindingsEvent,
        ]
        self.filter = BasicFilter(events_classes)
        self.social_broker = SocialBroker(tvm_dst_alias=config.get('social_broker_tvm_alias'))
        self.taxi = TaxiZalogin()

    def parse_message(self, message):
        return self.filter.filter(message)

    def process(self, header, data):
        message = MessageChunk(header, data)
        events = self.get_message_entries(message)
        for event in events:
            self.process_event(event)
        return True

    def process_event(self, event):
        if isinstance(event, SocialBindingsEvent):
            self.process_social_bindings_event(event)
        elif isinstance(event, BindPhonishAccountByTrackEvent):
            self.process_bind_phonish_account_by_track_event(event)
        else:
            raise NotImplementedError()

    def process_social_bindings_event(self, event):
        try:
            portal_uid = event.uid
            event_type = event.event_type
            for provider_code, provider_uid in event.providers_uids:
                self.taxi.uid_notify(
                    portal_uid=portal_uid,
                    phonish_uid=provider_uid,
                    event_type=event_type,
                )
                log.debug(
                    'Successfully notified Taxi. Uid=%s, event type=%s, provider code=%s, provider uid=%s',
                    portal_uid,
                    event_type,
                    provider_code,
                    provider_uid,
                )
        except BaseTaxiZaloginError as e:
            raise SocialBindingsHandlerException('Coundn\'t notify Taxi: %s' % e)

    def process_bind_phonish_account_by_track_event(self, event):
        try:
            response = self.social_broker.bind_phonish_account_by_track_v2(
                consumer=self.config.get('social_broker_consumer'),
                uid=event.uid,
                track_id=event.track_id,
                user_ip=event.ip,
            )
        except SocialBrokerProfileNotAllowedError:
            log.debug('Bind phonish to account %s not allowed (track_id=%s)' % (event.uid, event.track_id))
            return
        except BaseSocialBrokerError as e:
            log.error('Social broker failed: %s' % str(e))
            raise SocialBindingsHandlerException('Social broker failed: %s' % str(e))

        master_uid = response.get('uid')
        phonish_uid = response.get('phonish_uid')
        if not response.get('old'):
            log.info('Successfully bind phonish %s to account %s' % (phonish_uid, master_uid))
        else:
            log.info('Phonish %s is already bound to account %s' % (phonish_uid, master_uid))
