# -*- coding: utf-8 -*-
import json
import logging
import random
import time
import uuid

from cassandra import (
    ConsistencyLevel,
    OperationTimedOut,
    Timeout,
    Unavailable,
)
from cassandra.cluster import Cluster
from cassandra.policies import WhiteListRoundRobinPolicy
from passport.backend.core.ydb.exceptions import YdbTemporaryError
from passport.backend.core.ydb.ydb import YdbProfile
from passport.backend.library.historydbloader.historydb.utils import parse_comment
from passport.backend.logbroker_client.core.events.events import (
    AuthChallengeEvent,
    SessionKillEvent,
)
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.core.utils import is_yateam_uid


log = logging.getLogger(__name__)


class UfoHandlerException(HandlerException):
    pass


class MalformedUuidException(UfoHandlerException):
    pass


class UfoHandler(BaseHandler):
    handler_name = 'ufo'

    BLACKBOX_KEYSPACE = 'blackbox'
    INSERT_SESSION_TEMPLATE = 'INSERT INTO blackbox.ses_kills (authid, uid, ts) VALUES (?, ?, ?) USING TTL 7776000'

    PROFILE_KEYSPACE = 'profile'
    AUTH_ENV_TTL = 7 * 60 * 60 * 24
    INSERT_AUTH_ENV_TEMPLATE = 'INSERT INTO profile.profile (uid, id, data) VALUES (?, ?, ?) USING TTL %s' % AUTH_ENV_TTL
    INSERT_AUTH_ENV_RC_TEMPLATE = 'INSERT INTO profile.profile_rc (uid, id, data) VALUES (?, ?, ?) USING TTL %s' % AUTH_ENV_TTL

    def __init__(self, config, contact_points, consistency_level_name, **kwargs):
        super(UfoHandler, self).__init__(config=config, **kwargs)
        self.filter = BasicFilter([SessionKillEvent, AuthChallengeEvent])

        self.initialized = False
        self.cluster = None

        self.blackbox_session = None
        self.prepared_insert_session_query = None

        self.profile_session = None
        self.prepared_insert_auth_env_query = None

        self.contact_points = contact_points
        self.insert_session_consistency_level = ConsistencyLevel.name_to_value[consistency_level_name]

        self.ydb_clients = []

    def init_ydb_clients(self):
        self.ydb_clients = []
        for client_config in self.config['ydb']['clients']:
            self.ydb_clients.append(
                YdbProfile(
                    enabled=True,
                    endpoint=client_config['endpoint'],
                    database=client_config['database'],
                    table_name=client_config['table_name'],
                    auth_token=self.config['ydb']['token'],
                    retries=client_config['retries'],
                    timeout=client_config['timeout'],
                )
            )

    def _get_cassandra(self):
        return Cluster(self.contact_points, load_balancing_policy=WhiteListRoundRobinPolicy(self.contact_points))

    def init_cluster(self):
        if self.initialized:
            return

        self.cluster = self._get_cassandra()
        # Запись в таблицу blackbox.ses_kills
        self.blackbox_session = self.cluster.connect(self.BLACKBOX_KEYSPACE)
        self.prepared_insert_session_query = self.blackbox_session.prepare(self.INSERT_SESSION_TEMPLATE)
        self.prepared_insert_session_query.consistency_level = self.insert_session_consistency_level
        # Запись в таблицу profile.profile
        self.profile_session = self.cluster.connect(self.PROFILE_KEYSPACE)
        self.prepared_insert_auth_env_query = self.profile_session.prepare(self.INSERT_AUTH_ENV_TEMPLATE)
        self.prepared_insert_auth_env_query.consistency_level = self.insert_session_consistency_level
        self.prepared_insert_auth_env_rc_query = self.profile_session.prepare(self.INSERT_AUTH_ENV_RC_TEMPLATE)
        self.prepared_insert_auth_env_rc_query.consistency_level = self.insert_session_consistency_level

        self.init_ydb_clients()

        self.initialized = True

    def parse_message(self, message):
        return self.filter.filter(message)

    def process(self, header, data):
        self.init_cluster()

        message = MessageChunk(header, data)
        events = self.get_message_entries(message)

        self.process_events(events)
        return True

    def prepare_ydb_data(self, event):
        timestamp = int(event.timestamp) * 10 ** 6
        inverted_event_timestamp = self.get_inverted_timestamp(timestamp)
        unique_id = self.get_unique_id()

        ydb_env = dict(event.env)
        ydb_env.update({
            'inverted_event_timestamp': inverted_event_timestamp,
            'unique_id': unique_id,
        })
        ydb_key = {
            'uid': event.uid,
            'inverted_event_timestamp': inverted_event_timestamp,
            'unique_id': unique_id,
            'updated_at': timestamp,
        }
        ydb_payload = json.dumps(dict(
            timestamp=int(event.timestamp),
            **event.env
        ))
        return ydb_key, ydb_payload

    def get_timestamp(self):
        return int(time.time() * 1000 * 1000)

    def get_inverted_timestamp(self, timestamp=None):
        timestamp = timestamp or self.get_timestamp()
        return 2**64 - 1 - timestamp

    def get_unique_id(self):
        return random.getrandbits(64)

    def process_events(self, events):
        try:
            for event in events:
                if is_yateam_uid(event.uid):
                    continue
                if event.NAME == SessionKillEvent.NAME:
                    if not event.authid:
                        continue
                    self.blackbox_session.execute(
                        self.prepared_insert_session_query,
                        (event.authid, event.uid, int(event.timestamp)),
                    )
                elif event.NAME == AuthChallengeEvent.NAME:
                    comment = parse_comment(event.comment)
                    use_rc = comment.get('env') == 'rc'

                    payload = json.dumps(event.env)

                    try:
                        env_uuid = uuid.UUID(event.env_id)
                        if env_uuid.version != 1:
                            raise MalformedUuidException(json.dumps(event.to_dict()))
                    except ValueError:
                        raise MalformedUuidException(json.dumps(event.to_dict()))

                    self.profile_session.execute(
                        self.prepared_insert_auth_env_rc_query if use_rc else self.prepared_insert_auth_env_query,
                        (event.uid, env_uuid, payload),
                    )

                    ydb_key, ydb_payload = self.prepare_ydb_data(event)
                    for ydb_client in self.ydb_clients:
                        ydb_client.set(ydb_key, ydb_payload)
                        log.debug('Successfully written to YDB (%s): %s', ydb_client.endpoint, event.uid)
                else:
                    log.warning('Unknown event: %s', event)
        except MalformedUuidException as e:
            log.warning("Coundn't process event. %s", e)
        except (Unavailable, Timeout, OperationTimedOut) as e:
            raise UfoHandlerException("Coundn't write data to cassandra. %s: %s" % (type(e), e))
        except YdbTemporaryError as e:
            raise UfoHandlerException("Coundn't write data to YDB. %s: %s" % (type(e), e))
