# -*- coding: utf-8 -*-
import json
import logging
import time

from passport.backend.logbroker_client.core.events.events import (
    AccountDefaultAvatarEvent,
    AccountDisableEvent,
    AccountGlogoutEvent,
    AccountKarmaEvent,
    AppPasswordsRevokedEvent,
    SessionKillEvent,
    SessionsRevokedEvent,
    TokenInvalidateEvent,
)
from passport.backend.logbroker_client.core.events.filters import BasicFilter
from passport.backend.logbroker_client.core.handlers.base import BaseHandler
from passport.backend.logbroker_client.core.handlers.exceptions import HandlerException
from passport.backend.logbroker_client.core.handlers.utils import MessageChunk
from passport.backend.logbroker_client.core.monitoring import (
    MONITORING_STATUS_OK,
    MONITORING_STATUS_WARN,
    MonitoringState,
)
from passport.backend.logbroker_client.core.utils import (
    importobj,
    is_yateam_uid,
)
from passport.backend.logbroker_client.xiva.exceptions import (
    BaseXivaError,
    XivaPersistentError,
    XivaTemporaryError,
)
from passport.backend.logbroker_client.xiva.xiva import (
    Xiva,
    XivaHub,
)


log = logging.getLogger(__name__)


class MailHandlerException(HandlerException):
    pass


class MailHandler(BaseHandler):
    handler_name = 'xiva'

    ACCOUNT_INVALIDATE_NAME = 'account.invalidate'
    ACCOUNT_CHANGED_NAME = 'account.changed'
    SESSION_INVALIDATE_NAME = 'session.invalidate'
    CONNECTION_ID_FIELD = 'connection_id'

    def __init__(
        self, config, host, hub_host, stoken, stoken_stream, events,
        xiva=None, extra_host=None, yateam_host=None,
        **kwargs
    ):
        super(MailHandler, self).__init__(config=config, **kwargs)
        events_classes = [importobj(event) for event in events]
        self.filter = BasicFilter(events_classes)
        xiva_kwargs = dict(stoken=stoken, stoken_stream=stoken_stream, timeout=3)
        self.xiva = xiva if xiva else Xiva(host, **xiva_kwargs)
        self.extra_xiva = Xiva(extra_host, **xiva_kwargs) if extra_host else None
        self.yateam_xiva = Xiva(yateam_host, **xiva_kwargs) if yateam_host else None
        self.xiva_hub = XivaHub(hub_host, timeout=3)

    def parse_message(self, message):
        return self.filter.filter(message)

    def process(self, header, data):
        message = MessageChunk(header, data)
        events = self.get_message_entries(message)
        for event in events:
            self.process_event(event)
        return True

    def build_cnt_id_for_authid(self, authid):
        return 's:%s' % authid

    def build_cnt_id_for_token(self, token_id):
        return 't:%s' % token_id

    def event_mapper(self, event):
        """
        Возвращает признак необходимости отправки данных в потоковый сервис и словарь с данными для отправки
        """
        if event.name in (
            AccountDisableEvent.NAME,
            AccountGlogoutEvent.NAME,
            SessionsRevokedEvent.NAME,
            AppPasswordsRevokedEvent.NAME,
        ):
            return True, True, dict(
                name=self.ACCOUNT_INVALIDATE_NAME,
                timestamp=event.timestamp,
                comment=event.name,
            )
        elif event.name in (
            AccountKarmaEvent.NAME,
        ):
            return False, True, dict(
                name=self.ACCOUNT_CHANGED_NAME,
                timestamp=event.timestamp,
                comment=event.name,
                karma_status=event.karma_status,
                karma=event.karma,
            )
        elif event.name in (
            AccountDefaultAvatarEvent.NAME,
        ):
            # Изменения аватарок не отправляем в потоковый сервис
            return True, False, dict(
                name=self.ACCOUNT_CHANGED_NAME,
                timestamp=event.timestamp,
                comment=event.name,
            )
        elif event.name in (
            SessionKillEvent.NAME,
            TokenInvalidateEvent.NAME,
        ):
            result = dict(
                name=self.SESSION_INVALIDATE_NAME,
                timestamp=event.timestamp,
            )
            cntid = None
            if event.name == SessionKillEvent.NAME:
                cntid = self.build_cnt_id_for_authid(event.authid)
            elif event.name == TokenInvalidateEvent.NAME:
                cntid = self.build_cnt_id_for_token(event.token_id)
            if cntid:
                result[self.CONNECTION_ID_FIELD] = cntid
                return True, True, result

    def process_event(self, event):
        try:
            values = self.event_mapper(event)
            if values:
                is_usual_notification_required, is_stream_notification_required, mapped_event = values
                for handle, is_yateam_xiva in (
                        (self.xiva, False),
                        (self.extra_xiva, False),
                        (self.yateam_xiva, True),
                ):
                    if not handle:
                        continue
                    if is_yateam_xiva ^ is_yateam_uid(event.uid):
                        continue
                    if is_usual_notification_required:
                        handle.notify(
                            event.uid,
                            mapped_event['name'],
                            mapped_event['timestamp'],
                            json.dumps(mapped_event),
                        )
                    if is_stream_notification_required:
                        handle.stream_notify(
                            mapped_event['name'],
                            dict(mapped_event, uid=str(event.uid)),
                        )
                log.debug('Successfully notified Xiva. Uid=%s. Data=%s', event.uid, str(mapped_event))
        except XivaTemporaryError as e:
            raise MailHandlerException("Coundn't notify xiva. %s" % e)
        except XivaPersistentError as e:
            log.warning("Coundn't notify xiva. %s", e)
        except BaseXivaError as e:
            raise MailHandlerException("Coundn't notify xiva. %s" % e)

    def monitor(self, policy):
        with MonitoringState('xiva') as state:
            try:
                xiva_result = self.xiva_hub.ping()
                xiva_status = xiva_result == 'OK'
            except BaseXivaError:
                xiva_status = False

            now = time.time()
            first_unhealthy_ts = state['first_unhealthy_ts']
            last_unhealthy_ts = state['last_unhealthy_ts']
            unhealthy_cleanup_threshold = policy.get('max_unhealthy_cleanup_period', 60 * 60)
            unhealthy_grace_period = policy.get('unhealthy_grace_period', 60 * 60)

            if xiva_status:
                state['first_unhealthy_ts'] = state['last_unhealthy_ts'] = None
                unhealthy_period = None
                if last_unhealthy_ts:
                    unhealthy_period = min(last_unhealthy_ts - first_unhealthy_ts, unhealthy_cleanup_threshold)
                if not last_unhealthy_ts or now - last_unhealthy_ts > unhealthy_period / 2:
                    # Ксива работает, прошло достаточное количество времени с того момента, как она не работала, для
                    # выполнения последующих проверок
                    return False, None, None

                else:
                    # Ксива работает, но недавно была недоступна, поэтому пока пропускаем другие проверки
                    return True, MONITORING_STATUS_OK, 'xiva was unhealthy recently'

            else:
                if not first_unhealthy_ts:
                    state['first_unhealthy_ts'] = now
                state['last_unhealthy_ts'] = now

                if first_unhealthy_ts and now - first_unhealthy_ts > unhealthy_grace_period:
                    return True, MONITORING_STATUS_WARN, 'xiva is unhealthy for too long'

                return True, MONITORING_STATUS_OK, 'xiva is unhealthy'
