# -*- coding: utf-8 -*-
import json
import logging

from passport.backend.core.builders.base.base import BaseBuilder
from passport.backend.logbroker_client.xiva.exceptions import (
    BaseXivaError,
    XivaPersistentError,
    XivaRequestError,
    XivaTemporaryError,
)


log = logging.getLogger(__name__)


def check_response_for_errors(data, raw_response):
    if raw_response.status_code == 400:
        raise XivaPersistentError()
    elif raw_response.status_code != 200:
        raise XivaTemporaryError()


class Xiva(BaseBuilder):
    base_error_class = BaseXivaError
    temporary_error_class = XivaTemporaryError
    parser_error_class = XivaRequestError

    def __init__(self, url, stoken, stoken_stream, timeout, retries=3, useragent=None):
        self.stoken = stoken
        self.stoken_stream = stoken_stream
        super(Xiva, self).__init__(
            url=url,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
        )

    def notify(self, uid, event, event_ts, data=None):
        url = '/v1/send'
        params = {
            'stoken': self.stoken,
            'uid': uid,
            'event': event,
            'event_ts': event_ts,
        }
        return self._request_with_retries_simple(
            url_suffix=url,
            method='POST',
            error_detector=check_response_for_errors,
            parser=str,
            params=params,
            data=data,
        )

    def stream_notify(self, event, payload):
        url = '/v2/stream_send'
        params = {
            'token': self.stoken_stream,
            'event': event,
        }
        data = {
            'payload': payload,
        }
        return self._request_with_retries_simple(
            url_suffix=url,
            method='POST',
            error_detector=check_response_for_errors,
            parser=str,
            params=params,
            data=json.dumps(data),
        )


class XivaHub(BaseBuilder):
    base_error_class = BaseXivaError
    temporary_error_class = XivaTemporaryError
    parser_error_class = XivaRequestError

    def __init__(self, url, timeout, retries=3, useragent=None):
        super(XivaHub, self).__init__(
            url=url,
            timeout=timeout,
            retries=retries,
            logger=log,
            useragent=useragent,
        )

    def ping(self):
        url = '/notify/passport'
        params = {
            'operation': 'ping',
            'uid': 1,
        }
        return self._request_with_retries_simple(
            url_suffix=url,
            method='GET',
            error_detector=check_response_for_errors,
            parser=lambda response: response.content.strip(),
            params=params,
        )
