import logging
import os
import socket

from kazoo.client import (
    KazooClient,
    KazooState,
)
import luigi.process
import luigi.server


def state_listener(state):
    log = logging.getLogger(__name__)
    if state == KazooState.LOST or state == KazooState.SUSPENDED:
        log.info('luigid-python: session was lost or disconnected')
        pgrp = os.getpgid(os.getpid())
        os.killpg(pgrp, 9)
        exit()


def run_luigid(api_port, address, config):
    log = logging.getLogger(__name__)
    zk = KazooClient(hosts=','.join(config['zookeeper']['hosts']))
    zk.start()
    zk.add_listener(state_listener)
    zk.ensure_path(config['zookeeper']['lock_path'])
    zk.sync(config['zookeeper']['lock_path'])
    luigid_lock = zk.Lock(config['zookeeper']['lock_path'], socket.gethostname())
    luigid_lock.acquire(blocking=True)
    log.info('luigid starting...')
    luigi.configuration.get_config().set(
        'scheduler',
        'state_path',
        '/usr/lib/yandex/passport-luigid/state.pickle',
    )
    luigi.server.run(api_port=api_port, address=address)
    if zk.connected:
        luigid_lock.release()


def get_master_host(config):
    zk = KazooClient(hosts=','.join(config['zookeeper']['hosts']))
    zk.start()
    zk.add_listener(state_listener)
    zk.ensure_path(config['zookeeper']['lock_path'])
    zk.sync(config['zookeeper']['lock_path'])
    luigid_locks = zk.Lock(config['zookeeper']['lock_path']).contenders()
    if len(luigid_locks) < 1:
        raise RuntimeError('No running luigid')
    return luigid_locks[0]
