from passport.backend.library.nginx_config_generator import BaseConfig as Base


# Настройки внешних сервисов ------------------------------


_PASSPORT_HOSTS = {
    'localhost': {
        'testing': 'passport-test-internal.yandex.ru',
        'rc': 'passport-rc-internal.yandex.ru',
        'production': 'passport-internal.yandex.ru',
    },
    'intranet': {
        'testing': 'passport-test-internal.yandex-team.ru',
        'rc': 'passport-internal.yandex-team.ru',  # ятим-рц паспорта пока нет
        'production': 'passport-internal.yandex-team.ru',
    },
}

_OAUTH_HOSTS = {
    'localhost': {
        'testing': 'oauth-test-internal.yandex.ru',
        'rc': 'oauth-rc-internal.yandex.ru',
        'production': 'oauth-internal.yandex.ru',
    },
    'intranet': {
        'testing': 'oauth-test-internal.yandex-team.ru',
        'production': 'oauth-internal.yandex-team.ru',
    },
}

_YANDEX_LOGIN_HOSTS = {
    'localhost': {
        'testing': 'login-test.yandex.ru',
        'rc': 'login-rc.yandex.ru',
        'production': 'login.yandex.ru',
    },
    'intranet': {
        'testing': 'login-test.yandex-team.ru',
        'production': 'login.yandex-team.ru',
    },
}


_MAIL_HOSTS = {
    'localhost': {
        'testing': 'xeno.mail.yandex.net',
        'rc': 'xeno.mail.yandex.net',
        'production': 'xeno.mail.yandex.net',
    },
    'intranet': {
        'testing': '',
        'production': '',
    },
}


_SOCIAL_HOSTS = {
    'localhost': {
        'testing': 'social-test.yandex.ru',
        'rc': 'social-rc.yandex.ru',
        'production': 'social.yandex.ru',
    },
    'intranet': {
        'testing': '',
        'production': '',
    },
}


_SOCIAL_API_HOSTS = {
    'localhost': {
        'testing': 'api.social-test.yandex.ru',
        'rc': 'social-rc.yandex.ru',
        'production': 'api.social.yandex.ru',
    },
    'intranet': {
        'testing': '',
        'production': '',
    },
}


_UAAS_PROXY_HOSTS = {
    'localhost': {
        'testing': 'uaas-test.passport.yandex.net',
        'rc': 'uaas.passport.yandex.net',
        'production': 'uaas.passport.yandex.net',
    },
    'intranet': {
        'testing': '',
        'production': '',
    },
}


_PUSH_SUBSCRIPTION_HOSTS = {
    'localhost': {
        'testing': 'push-subscription-test.passport.yandex.net',
        'rc': 'push-subscription-rc.passport.yandex.net',
        'production': 'push-subscription.passport.yandex.net',
    },
    'intranet': {
        'testing': '',
        'production': '',
    },
}


# Базовые конфиги -------------------------------------------------------------


class BaseConfig(Base):
    _tunnel_ips = []
    _tunnel_only = False  # не подниматься, если нет активных туннелей

    def __init__(self):
        super(BaseConfig, self).__init__()
        if self.yenv_type == 'development':
            self.yenv_type = 'testing'

    @property
    def is_applicable(self):
        if not super(BaseConfig, self).is_applicable:
            return False
        # Если работаем только под туннелем, а туннеля нет - значит, в этом окружении подниматься не надо
        if self._tunnel_only and not self._get_tunnel_ips():
            return False
        return True

    def _get_tunnel_ips(self):
        """Возвращает IP-адреса туннелей, ведущих в данную машину"""
        return self._tunnel_ips if self.is_tunnel_enabled else []

    @property
    def is_tunnel_enabled(self):
        return bool(self._tunnel_ips)

    def get_custom_settings(self):
        passport_host = _PASSPORT_HOSTS[self.yenv_name][self.yenv_type]
        passport_yateam_host = _PASSPORT_HOSTS['intranet'][self.yenv_type]
        oauth_host = _OAUTH_HOSTS[self.yenv_name][self.yenv_type]
        yandex_login_host = _YANDEX_LOGIN_HOSTS[self.yenv_name][self.yenv_type]
        mail_host = _MAIL_HOSTS[self.yenv_name][self.yenv_type]
        social_host = _SOCIAL_HOSTS[self.yenv_name][self.yenv_type]
        social_api_host = _SOCIAL_API_HOSTS[self.yenv_name][self.yenv_type]
        uaas_proxy_host = _UAAS_PROXY_HOSTS[self.yenv_name][self.yenv_type]
        push_subscription_host = _PUSH_SUBSCRIPTION_HOSTS[self.yenv_name][self.yenv_type]

        return dict(
            super(BaseConfig, self).get_custom_settings(),
            tunnel_ips=self._get_tunnel_ips(),
            passport_host=passport_host,  # хост паспорта, куда проксируем
            passport_yateam_host=passport_yateam_host,  # хост паспорта для ручки user_info/intranet
            oauth_host=oauth_host,  # хост OAuth, куда проксируем
            yandex_login_host=yandex_login_host,  # хост Я.Логина, куда проксируем
            mail_host=mail_host,  # хост Почты, куда проксируем
            social_host=social_host,  # хост Социализма (фронтенд), куда проксируем
            social_api_host=social_api_host,  # хост Социализма (апи), куда проксируем
            uaas_proxy_host=uaas_proxy_host,  # хост прокси, ходящей в uaas
            push_subscription_host=push_subscription_host,  # хост апи для подписки на пуши
        )


class BaseMobileproxyConfig(BaseConfig):
    _slb_ips = {
        'localhost': {
            'testing': (
                '213.180.204.115',
                '2a02:6b8::1:115',
            ),
            'rc': (
                '2a02:6b8:0:3400::933',
            ),
            'production': (
                '213.180.193.115',
                '2a02:6b8::115',
            ),
        },
    }
    _tunnel_ips = (
        '5.45.202.34',
        '5.45.202.35',
        '5.45.202.36',
        '5.45.202.37',
    )

    @property
    def is_tunnel_enabled(self):
        return (
            super(BaseMobileproxyConfig, self).is_tunnel_enabled and
            self.yenv_type == 'production' and self.fqdn.startswith('mobileproxy-')
        )

    @property
    def is_applicable(self):
        # В тестинге везде, в проде - только на выделенных машинах
        return (
            super(BaseMobileproxyConfig, self).is_applicable and
            (self.yenv_type != 'production' or self.fqdn.startswith('mobileproxy-'))
        )

    @property
    def use_antirobot(self):
        return True

    def get_custom_settings(self):
        return dict(
            super(BaseMobileproxyConfig, self).get_custom_settings(),
            use_antirobot=self.use_antirobot,
        )


class BaseCertificateConfig(BaseConfig):
    _slb_ips = {
        'localhost': {
            'testing': (
                '93.158.134.115',
                '2a02:6b8::2:115',
            ),
            'production': (
                # certificate.mobile.yandex.net
                '87.250.250.115',
                '2a02:6b8::3:115',
                # certificate-partners.mobile.yandex.net
                '87.250.250.245',
                '2a02:6b8::3:245',
            ),
        },
    }
    _tunnel_ips = (
        '5.45.202.62',
        '5.45.202.63',
        '5.45.202.64',
        '5.45.202.65',
    )

    @property
    def is_tunnel_enabled(self):
        return (
            super(BaseCertificateConfig, self).is_tunnel_enabled and
            self.yenv_type == 'production' and self.fqdn.startswith('certificate-')
        )

    @property
    def is_applicable(self):
        # В тестинге везде, в проде - только на выделенных машинах и в Финке
        return (
            super(BaseCertificateConfig, self).is_applicable and
            (self.yenv_type != 'production' or self.fqdn.startswith('certificate-'))
        )


# Конфиги mobileproxy ---------------------------------------------------------


class MobileproxyAll(BaseMobileproxyConfig):
    _cert_names = {
        'localhost': {
            'testing': 'mobileproxy-test.passport.yandex.net',
            'rc': 'mobileproxy-rc.passport.yandex.net',
            'production': 'mobileproxy.passport.yandex.net',
        },
    }
    _server_names = {
        'localhost': {
            'testing': [
                'mobileproxy-test.passport.yandex.net',
            ],
            'rc': [
                'mobileproxy-rc.passport.yandex.net',
            ],
            'production': [
                'mobileproxy.passport.yandex.net',
            ],
        },
    }


class MobileproxyInternal(BaseMobileproxyConfig):
    _ssl_required = False

    _server_names = {
        'localhost': {
            'testing': ['mobileproxy-test-internal.passport.yandex.net'],
            'rc': ['mobileproxy-rc-internal.passport.yandex.net'],
            'production': ['mobileproxy-internal.passport.yandex.net'],
        },
    }

    @property
    def is_applicable(self):
        return super(MobileproxyInternal, self).is_applicable and self.yenv_name == 'localhost'


class MobileproxyYateam(BaseMobileproxyConfig):
    def __init__(self):
        super(MobileproxyYateam, self).__init__()
        self.yenv_name = 'intranet'
        self._slb_ips['intranet'] = self._slb_ips['localhost']

    _cert_names = {
        'intranet': {
            'testing': 'mobileproxy-test.passport.yandex.net',
            'production': 'mobileproxy.passport.yandex.net',
        },
    }
    _server_names = {
        'intranet': {
            'testing': [
                'mobileproxy-yateam-test.passport.yandex.net',
            ],
            'production': [
                'mobileproxy-yateam.passport.yandex.net',
            ],
        },
    }


class MobileproxyRegistrator(BaseMobileproxyConfig):
    _cert_names = {
        'localhost': {
            'testing': 'mobileproxy-test.passport.yandex.net',
            'production': 'mobileproxy.passport.yandex.net',
        },
    }
    _server_names = {
        'localhost': {
            'testing': [
                'registrator.tst.mobile.yandex.net',
            ],
            'production': [
                'registrator.mobile.yandex.net',
                'mobile-partners.registrator.mobile.yandex.net',
                'megafon.mobile-partners.registrator.mobile.yandex.net',
            ],
        },
    }


class MobileproxyOauth(BaseMobileproxyConfig):
    _cert_names = {
        'localhost': {
            'testing': 'mobileproxy-test.passport.yandex.net',
            'production': 'mobileproxy.passport.yandex.net',
        },
    }
    _server_names = {
        'localhost': {
            'testing': [
                'oauth.tst.mobile.yandex.net',
            ],
            'production': [
                'oauth.mobile.yandex.net',
                'oauth.mobile.yandex.net:443',
                'oauth.mobile-partners.yandex.net',
                'megafon.oauth.mobile-partners.yandex.net',
            ],
        },
    }


class MobileproxyAmazonAll(BaseMobileproxyConfig):
    _cert_names = {
        'localhost': {
            'production': 'mobileproxy.mobile.pssp.z5h64q92x9.net',
        },
    }
    _server_names = {
        'localhost': {
            'production': [
                'mobileproxy.mobile.pssp.z5h64q92x9.net',
            ],
        },
    }


class MobileproxyAmazonSmilink(BaseMobileproxyConfig):
    _tunnel_only = True
    _cert_names = {
        'localhost': {
            'production': 'mobileproxy.mobile.pssp.smilink.ru',
        },
    }
    _server_names = {
        'localhost': {
            'production': [
                'mobileproxy.mobile.pssp.smilink.ru',
            ],
        },
    }


# Конфиги certificate ---------------------------------------------------------


class MobileproxyCertificate(BaseCertificateConfig):
    _cert_names = {
        'localhost': {
            'testing': 'certificate.tst.mobile.yandex.net',
            'production': 'certificate.mobile.yandex.net',
        },
    }
    _server_names = {
        'localhost': {
            'testing': [
                'certificate.tst.mobile.yandex.net',
            ],
            'production': [
                'certificate.mobile.yandex.net',
                'certificate.mobile.yandex.net:443',
                'certificate-partners.mobile.yandex.net',
                'certificate-pin.yandex.net',
            ],
        },
    }


class MobileproxyAmazonCertificate(BaseCertificateConfig):
    _cert_names = {
        'localhost': {
            'production': 'certificate.mobile.pssp.z5h64q92x9.net',
        },
    }
    _server_names = {
        'localhost': {
            'production': [
                'certificate.mobile.pssp.z5h64q92x9.net',
            ],
        },
    }

# -----------------------------------------------------------------------------


CONFIG_CLASSES_WITH_PRIORITIES = {
    0: MobileproxyAll,  # всегда первый
    10: MobileproxyRegistrator,
    11: MobileproxyOauth,
    12: MobileproxyCertificate,
    13: MobileproxyInternal,
    20: MobileproxyYateam,
    80: MobileproxyAmazonAll,
    81: MobileproxyAmazonCertificate,
    82: MobileproxyAmazonSmilink,
}
