# -*- coding: utf-8 -*-
from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError
from passport.backend.oauth.admin.admin.utils import get_resolve_blackbox
from passport.backend.oauth.core.common.blackbox_helpers import get_uid_by_login_or_uid


DEFAULT_TOKEN_LENGTH = 39


class UidField(forms.fields.CharField):
    def clean(self, value):
        value = super(UidField, self).clean(value)
        uid = clean_user(value)
        if uid is not None:
            return uid
        raise ValidationError('Пользователь с логином или уидом "%s" не найден' % value)


class CollectionField(forms.fields.Field):
    def __init__(self, validator, max_count, readonly=False, **kwargs):
        widget_attrs = {'rows': 2, 'cols': 32}
        if readonly:
            widget_attrs.update(readonly='readonly')
        super(CollectionField, self).__init__(
            required=False,
            widget=forms.Textarea(attrs=widget_attrs),
            **kwargs
        )
        self._validator = validator
        self._max_count = max_count

    def clean(self, value):
        raw_value = super(CollectionField, self).clean(value).strip().split()
        if not raw_value:
            return []
        if len(raw_value) > self._max_count:
            raise ValidationError('Слишком много значений')
        return [self._validator.clean(item) for item in raw_value]


class ClientForm(forms.Form):
    client_id = forms.fields.CharField(required=True, max_length=32, widget=forms.widgets.HiddenInput())


class EditClientForm(forms.Form):
    id = forms.fields.CharField(
        label='ID',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    secret = forms.fields.CharField(
        label='Secret',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    default_title = forms.fields.CharField(
        label='Название (default)',
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    title_ru = forms.fields.CharField(
        label='Название (ru)',
        required=False,
    )
    title_uk = forms.fields.CharField(
        label='Название (uk)',
        required=False,
    )
    title_en = forms.fields.CharField(
        label='Название (en)',
        required=False,
    )
    title_tr = forms.fields.CharField(
        label='Название (tr)',
        required=False,
    )
    default_description = forms.fields.CharField(
        label='Описание (default)',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    description_ru = forms.fields.CharField(
        label='Описание (ru)',
        required=False,
    )
    description_uk = forms.fields.CharField(
        label='Описание (uk)',
        required=False,
    )
    description_en = forms.fields.CharField(
        label='Описание (en)',
        required=False,
    )
    description_tr = forms.fields.CharField(
        label='Описание (tr)',
        required=False,
    )
    token_ttl = forms.fields.CharField(
        label='Срок жизни токена',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    scopes = forms.fields.CharField(
        label='Права',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    deleted_scopes = forms.fields.CharField(
        label='Устаревшие права',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    extra_visible_scopes = forms.fields.CharField(
        label='Скрытые права, доступные приложению',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    homepage = forms.fields.URLField(
        label='Ссылка на приложение',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    redirect_uris = CollectionField(
        validator=forms.fields.CharField(),
        max_count=20,
        readonly=True,
    )
    approval_status = forms.fields.CharField(
        label='Модерация',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    created = forms.fields.CharField(
        label='Дата создания',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    modified = forms.fields.CharField(
        label='Дата последнего изменения',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    glogouted = forms.fields.CharField(
        label='Дата последнего отзыва токенов',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    deleted = forms.fields.CharField(
        label='Дата удаления',
        required=False,
        widget=forms.TextInput(attrs={'readonly': True}),
    )
    creator = UidField(
        label='Создатель (uid или логин)',
        max_length=50,
        required=False,
    )
    owner_uids = CollectionField(
        validator=UidField(),
        max_count=5,
        label='Владельцы (uid\'ы или логины)',
    )
    owner_groups = CollectionField(
        validator=forms.fields.CharField(max_length=50),
        max_count=5,
        label='Группы владельцев',
    )
    owner_group_logins = CollectionField(
        validator=forms.fields.CharField(max_length=50),
        max_count=300,
        readonly=True,
        label='Юзеры из групп (автообновлённые)',
    )
    is_yandex = forms.fields.BooleanField(
        label='Приложение является яндексовым',
        required=False,
    )
    telegram_bot_name = forms.fields.CharField(
        label='Имя связанного Telegram-бота',
        max_length=100,
        required=False,
    )
    ios_default_app_id = forms.fields.CharField(
        label='AppID для iOS SDK (основной)',
        max_length=100,
        required=False,
    )
    ios_extra_app_ids = forms.fields.CharField(
        label='AppID для iOS SDK (дополнительные)',
        required=False,
        widget=forms.widgets.Textarea(attrs={'rows': 2, 'cols': 32}),
    )
    ios_appstore_url = forms.fields.CharField(
        label='Урл приложения в AppStore',
        max_length=200,
        required=False,
    )
    android_default_package_name = forms.fields.CharField(
        label='PackageName для Android SDK (основной)',
        max_length=100,
        required=False,
    )
    android_extra_package_names = forms.fields.CharField(
        label='PackageName для Android SDK (дополнительные)',
        required=False,
        widget=forms.widgets.Textarea(attrs={'rows': 2, 'cols': 32}),
    )
    android_cert_fingerprints = forms.fields.CharField(
        label='Отпечатки для Android SDK',
        required=False,
        widget=forms.widgets.Textarea(attrs={'rows': 2, 'cols': 32}),
    )
    android_appstore_url = forms.fields.CharField(
        label='Урл приложения в Google Play',
        max_length=200,
        required=False,
    )
    turboapp_base_url = forms.fields.CharField(
        label='Base_url турбоаппа',
        max_length=200,
        required=False,
    )
    allow_nonpublic_granttypes = forms.fields.BooleanField(
        label='Разрешёны непубличные grant_type',
        required=False,
    )

    def clean_ios_extra_app_ids(self):
        raw_value = self.cleaned_data['ios_extra_app_ids'].strip()
        return [item.strip() for item in raw_value.split()]

    def clean_android_extra_package_names(self):
        raw_value = self.cleaned_data['android_extra_package_names'].strip()
        return [item.strip() for item in raw_value.split()]

    def clean_android_cert_fingerprints(self):
        raw_value = self.cleaned_data['android_cert_fingerprints'].strip()
        return [item.strip() for item in raw_value.split()]


def clean_user(cleaned_data):
    return get_uid_by_login_or_uid(
        cleaned_data.strip(),
        blackbox=get_resolve_blackbox(),
    )


class ScopeForm(forms.Form):
    keyword = forms.fields.CharField(
        label='Кейворд',
        required=True,
        max_length=100,
    )
    default_title = forms.fields.CharField(
        label='Человекочитаемое название',
        required=True,
        max_length=250,
        widget=forms.TextInput(attrs={'size': 40}),
    )
    ttl = forms.fields.IntegerField(
        label='Время жизни (TTL) токена (в секундах)',
        required=False,
        min_value=0,
    )
    is_ttl_refreshable = forms.fields.BooleanField(
        label='Подновлять TTL при использовании токена',
        required=False,
    )
    requires_approval = forms.fields.BooleanField(
        label='Требует премодерации',
        required=False,
    )
    is_hidden = forms.fields.BooleanField(
        label='Не показывать на странице регистрации приложения',
        required=False,
    )
    visible_for_uids = CollectionField(
        validator=UidField(),
        max_count=20,
        label='Право видно следующим людям (uid или login, разделённые пробелами или \\n)',
    )
    has_xtoken_grant = forms.fields.BooleanField(
        label='Токен с данным скоупом является Х-токеном',
        required=False,
    )

    def clean_keyword(self):
        FORBIDDEN_SERVICE_NAMES = ['grant_type']
        keyword = self.cleaned_data['keyword'].strip()
        if ':' not in keyword.strip(':'):
            self.add_error('keyword', 'Не указано имя сервиса')
        if any(keyword.startswith('%s:' % stop_word) for stop_word in FORBIDDEN_SERVICE_NAMES):
            self.add_error('keyword', 'Недопустимое имя сервиса')
        return keyword

    def clean(self):
        cleaned_data = super(ScopeForm, self).clean()
        if cleaned_data['is_ttl_refreshable']:
            cleaned_data['ttl'] = settings.DEFAULT_TOKEN_TTL
        return cleaned_data


class FindUserForm(forms.Form):
    uid = UidField(label='Логин или uid')


class EditUserForm(forms.Form):
    is_corporate = forms.fields.BooleanField(
        label='Может создавать яндексовые приложения',
        required=False,
    )


class VerifyTokenForm(forms.Form):
    type = forms.ChoiceField(
        choices=(
            ('access_token', 'access_token'),
            ('access_token_prefix', 'access_token_prefix'),
            ('refresh_token', 'refresh_token'),
            ('uid:app_password', 'uid:app_password'),
            ('token_id', 'token_id'),
        ),
        initial='access_token',
        required=False,
    )
    credential_to_search = forms.CharField(
        label='Поиск',
        required=False,
        widget=forms.TextInput(attrs={'placeholder': 'Введите запрос'}),
    )

    def clean(self):
        self.cleaned_data.update(
            token_id=None,
            access_token=None,
            refresh_token=None,
            uid=None,
            app_password=None,
        )
        type_ = self.cleaned_data['type']
        search = self.cleaned_data['credential_to_search']

        if not search:
            raise ValidationError('No search request')
        elif type_ == 'token_id' and search.isdigit():
            self.cleaned_data['token_id'] = int(search)
        elif type_ == 'access_token':
            self.cleaned_data['access_token'] = search
        elif type_ == 'access_token_prefix':
            self.cleaned_data['access_token'] = search.ljust(DEFAULT_TOKEN_LENGTH, '*')
            self.cleaned_data['type'] = 'access_token'
        elif type_ == 'refresh_token':
            self.cleaned_data['refresh_token'] = search
        elif type_ == 'uid:app_password' and ':' in search:
            self.cleaned_data['uid'], _, self.cleaned_data['token_alias'] = search.partition(':')
        else:
            raise ValidationError('Invalid form')
        return self.cleaned_data


class EncryptAmCredsForm(forms.Form):
    decrypted = forms.CharField(
        required=False,
        widget=forms.widgets.Textarea(attrs={
            'rows': 3,
            'cols': 60,
            'placeholder': 'Сырые значения',
        }),
    )
    encrypted = forms.CharField(
        required=False,
        widget=forms.widgets.Textarea(attrs={
            'rows': 3,
            'cols': 80,
            'placeholder': 'Зашифрованные значения',
        }),
    )


class SearchForm(forms.Form):
    def __init__(self, services, **kwargs):
        super(SearchForm, self).__init__(**kwargs)
        self.fields['service'].choices = (
            ('', '<все доступные сервисы>'),
        ) + tuple(
            sorted(zip(services, services)),
        )

    service = forms.ChoiceField(
        choices=(),
        initial='',
        required=False,
    )
    requires_approval = forms.fields.BooleanField(
        label='Только требующие премодерации',
        required=False,
    )
    yandex_only = forms.fields.BooleanField(
        label='Только яндексовые приложения',
        required=False,
    )
    search = forms.CharField(
        label='Поиск',
        required=False,
        widget=forms.TextInput(attrs={'placeholder': 'Введите поисковый запрос'}),
    )
    search_type = forms.ChoiceField(
        choices=(
            ('client_id', 'client_id'),
            ('client_id_internal', 'client_id_internal'),
            ('creator_uid', 'creator_uid'),
            ('creator_login', 'creator_login'),
        ),
        initial='client_id',
        required=False,
    )

    def clean(self):
        cleaned_data = super(SearchForm, self).clean()
        if cleaned_data['search_type'] == 'creator_login' and self.cleaned_data['search']:
            resolved_uid = clean_user(self.cleaned_data['search'])
            if resolved_uid is not None:
                cleaned_data['search'] = resolved_uid
                cleaned_data['search_type'] = 'creator_uid'
        return cleaned_data


class FilterForm(forms.Form):
    def __init__(self, scopes, **kwargs):
        super(FilterForm, self).__init__(**kwargs)
        self.fields['scope'].choices = (
            ('', '<все доступные скоупы>'),
        ) + tuple(
            sorted((s.keyword, s.keyword) for s in scopes),
        )

    scope = forms.ChoiceField(
        choices=(),
        initial='',
        required=False,
    )
    title = forms.CharField(
        label='Название содержит',
        required=False,
        widget=forms.TextInput(),
    )
    only_with_nonpublic_granttypes = forms.fields.BooleanField(
        label='Только с непубличными grant_type',
        required=False,
    )
